/*-------------------------------------------------------------------------*/
/*  J3W    3D Animation Kit  (compiler)                                    */
/*  j3c_asm.cpp    08/06/2012                                              */
/*  Copyright (C) 1998 - 2012 Jun Mizutani <mizutani.jun@nifty.ne.jp>      */
/*                      All rights reserved.                               */
/*                                                                         */
/*   This file is part of the J3W 3D Animation Kit, and is covered under   */
/*  the terms of the GNU General Public License, version 2. This file has  */
/*  NO WARRANTY. See file COPYING for copyright details.                   */
/*                                                                         */
/*-------------------------------------------------------------------------*/

#include <string>
#include <iomanip>
#include <stdio.h>

using namespace std;

#include "j3c_word.h"
#include "j3c_asm.h"

const int BUF_SIZE = 16;
const int EXP_NEST = 16;

cAsmFile::cAsmFile(ostream &os) : dist(os), factor_count(0)
{
    fcount = new cStack(EXP_NEST);
}

cAsmFile::~cAsmFile()
{
    delete fcount;
}

void cAsmFile::eol()
{
    dist << endl;
}

void cAsmFile::label(string lb)
{
    dist << lb << endl;
}

void cAsmFile::opcode(string op, string operand)
{
    dist << "        " << op << operand << endl;
}

void cAsmFile::comment(string comment)
{
    dist << "        ;" << comment << endl;
}

void cAsmFile::jump(string label)
{
    opcode("BRA     ", label);
}

void cAsmFile::begin(string name)
{
    label(name + ":");
    opcode("CALL    ", name + "_INIT:");
    opcode("CALL    ", name + "_RUN:");
    opcode("DELPRC  ", "");
}

void cAsmFile::init_begin(string name)
{
    label(name + "_INIT:");
}

void cAsmFile::init_end()
{
    opcode("RETURN", "");
}

void cAsmFile::method_begin(string name)
{
    label(name);
}

void cAsmFile::method_end()
{
    opcode("LOAD    ","R4  R1");
    opcode("RETURN", "");
    dist << endl;
}

void cAsmFile::run_begin(string name)
{
    label(name + "_RUN:");
    opcode("RECEIV  ", "RL");
    opcode("CMP     ", "RL  0");
    opcode("BEQ     ", name + "_RUN_BODY:");
    opcode("CALL    ", name + "_EVENT:");
    label(name + "_RUN_BODY:");
}

void cAsmFile::run_end(string name)
{
    opcode("THROW   ", "");
    opcode("BRA     ", name + "_RUN:");
}

void cAsmFile::event_begin(string name)
{
    label(name + "_EVENT:");
}

void cAsmFile::event_end(string name)
{
    opcode("THROW   ", "");
    opcode("RETURN  ", "");

}

/*----------------------------------------------------------------*/
/* EXPRESSION                                                     */
/*----------------------------------------------------------------*/
void cAsmFile::exp1()
{
    switch(optimize) {
      case 0 : opcode("LOAD    ","R1  R2"); break;
      case 1 : opcode("LOAD    ","R1  R3"); optimize = 0; break;
      case 2 : opcode("LOAD    ","R1  R4"); optimize = 0; break;
    }
}

void cAsmFile::exp2(int sym, string lb1, string lb2)
{
    switch(optimize) {
      case 0 : opcode("CMP     ","R1  R2"); break;
      case 1 : opcode("CMP     ","R1  R3"); optimize = 0; break;
      case 2 : opcode("CMP     ","R1  R4"); optimize = 0; break;
    }
    switch(sym) {
        case sym_2EQL :
             opcode("BEQ     ", lb1);
             break;
        case sym_GTR :
             opcode("BGR     ", lb1);
             break;
        case sym_LESS :
             opcode("BLS     ", lb1);
             break;
        case sym_GTREQL:
             opcode("BGE     ", lb1);
             break;
        case sym_LESSEQL:
             opcode("BLE     ", lb1);
             break;
        case sym_EXEQL:
             opcode("BNE     ", lb1);
             break;
    }
    opcode("LOAD    ", "R1  0");
    opcode("BRA     ", lb2);
    label(lb1);
    opcode("LOAD    ", "R1  -1");
    label(lb2);
}

void cAsmFile::simple_exp1()
{
    switch(optimize) {
      case 0 : opcode("LOAD    ","R2  R3"); break;
      case 1 : opcode("LOAD    ","R2  R4"); optimize = 0; break;
    }
}

void cAsmFile::simple_exp2(int sym)
{
    switch(sym) {
        case sym_PLUS :
             if (optimize == 1) {
                 opcode("ADD     ", "R2  R4");
                 optimize = 0;
             } else opcode("ADD     ", "R2  R3");
             break;
        case sym_MINUS :
             if (optimize == 1) {
                 opcode("SUB     ", "R2  R4");
                 optimize = 0;
             } else opcode("SUB     ", "R2  R3");
             break;
        case sym_BAR :
             if (optimize == 1) {
                 opcode("OR      ", "R2  R4");
                 optimize = 0;
             } else opcode("OR      ", "R2  R3");
             break;
    }
}

void cAsmFile::term1()
{
    opcode("LOAD    ","R3  R4");
}

void cAsmFile::term2(int sym)
{
    switch(sym) {
        case sym_MUL :
             opcode("MUL     ", "R3  R4");
             break;
        case sym_SLASH :
             opcode("DIV     ", "R3  R4");
             break;
        case sym_AMP :
             opcode("AND     ", "R3  R4");
             break;
    }
}

void cAsmFile::paren1()
{
    opcode("PUSH    ","R1");
    opcode("PUSH    ","R2");
    opcode("PUSH    ","R3");
}

void cAsmFile::paren2()
{
    opcode("LOAD    ","R4  R1");
    opcode("POP     ","R3");
    opcode("POP     ","R2");
    opcode("POP     ","R1");
}

void cAsmFile::minus()
{
    opcode("NEG     ","R4  R4");
}

void cAsmFile::not0()
{
    opcode("NOT     ","R4  R4");
}

void cAsmFile::character(char ch)
{
    char v[BUF_SIZE];
    sprintf(v, "R4  '%c'", ch);
    opcode("LOAD    ", v);
    opcode("LOAD    ","R4");
}

void cAsmFile::var_ref(int addr)
{
    char v[BUF_SIZE];
    sprintf(v, "R4  %d#", addr);
    opcode("LOAD    ", v);
}

void cAsmFile::array_ref(int addr)
{
    char v[BUF_SIZE];
    sprintf(v, "R4  %d", addr);
    opcode("LOAD    ", v);
}

void cAsmFile::param_ref(int pos)
{
    char v[BUF_SIZE];
    sprintf(v, "R4  %d#", pos);
    opcode("LOADBP  ", v);
}

void cAsmFile::array_ref1()
{
    opcode("PUSH    ","R1");
    opcode("PUSH    ","R2");
    opcode("PUSH    ","R3");
}

void cAsmFile::array_ref2(int addr)
{
    char v[BUF_SIZE];
    sprintf(v, "R1  %d", addr);
    opcode("ADD     ", v);
    opcode("XLOAD   ","R4  R1");
    opcode("POP     ","R3");
    opcode("POP     ","R2");
    opcode("POP     ","R1");
}

void cAsmFile::number(int num)
{
    char v[BUF_SIZE];
    sprintf(v, "R4  %d", num);
    opcode("LOAD    ", v);
}

/*----------------------------------------------------------------*/
/* ASSIGNMENT                                                     */
/*----------------------------------------------------------------*/
void cAsmFile::assign(int value)
{
    char v[BUF_SIZE];
    sprintf(v, "R1  %d#", value);
    opcode("STORE   ", v);
}

void cAsmFile::assign_param(int value)
{
    char v[BUF_SIZE];
    sprintf(v, "R1  %d#", value);
    opcode("STORBP  ", v);
}

void cAsmFile::assign_array1()
{
    opcode("PUSH    ","R1");
}

void cAsmFile::assign_array2(int addr)
{
    char v[BUF_SIZE];

    opcode("POP     ","R2");
    sprintf(v, "R2  %d", addr);
    opcode("ADD     ", v);
    opcode("XSTORE  ", "R1  R2");
}

/*----------------------------------------------------------------*/
/* IF                                                             */
/*----------------------------------------------------------------*/
void cAsmFile::if1(string lb1)
{
    opcode("CMP     ", "R1 0");
    opcode("BEQ     ", lb1);
}
void cAsmFile::if2(string lb1, string lb2)
{
    opcode("BRA     ", lb2);
    label(lb1);
}
void cAsmFile::if3(string lb2)
{
    label(lb2);
}

/*----------------------------------------------------------------*/
/* WHILE                                                          */
/*----------------------------------------------------------------*/
void cAsmFile::while1(string lb1)
{
    label(lb1);
}

void cAsmFile::while2(string lb2)
{
    opcode("CMP     ", "R1  0");
    opcode("BEQ     ", lb2);
}

void cAsmFile::while3(string lb1, string lb2)
{
    opcode("BRA     ", lb1);
    label(lb2);
}

/*----------------------------------------------------------------*/
/* DO                                                             */
/*----------------------------------------------------------------*/
void cAsmFile::do1(string lb1)
{
    label(lb1);
}

void cAsmFile::do2(string lb1, string lb2)
{
    opcode("CMP     ", "R1  0");
    opcode("BNE     ", lb1);
}

/*----------------------------------------------------------------*/
/* SWITCH                                                         */
/*----------------------------------------------------------------*/
void cAsmFile::switch1()
{
    opcode("PUSH    ", "R1");
}

void cAsmFile::switch2(string lb1, string lb2, string lb3, int val)
{
    char v[BUF_SIZE];
    label(lb1);
    opcode("POP     ", "R1");
    opcode("PUSH    ", "R1");
    sprintf(v, "R1  %d", val);
    opcode("CMP     ", v);
    opcode("BNE     ", lb2);
    label(lb3);
}

void cAsmFile::switch3(string lb4)
{
    label(lb4);
    opcode("POP     ", "R1");
}

void cAsmFile::switch4(string lb5)
{
    label(lb5);
}

void cAsmFile::switch5(string lb6)
{
    opcode("BRA     ", lb6);
}

/*----------------------------------------------------------------*/
/* RETURN                                                         */
/*----------------------------------------------------------------*/
void cAsmFile::return0()
{

}

/*----------------------------------------------------------------*/
/* BREAK                                                          */
/*----------------------------------------------------------------*/
void cAsmFile::break0(string lb)
{
    opcode("BRA     ", lb);
}

/*----------------------------------------------------------------*/
/* CONTINUE                                                       */
/*----------------------------------------------------------------*/
void cAsmFile::continue0(string lb)
{
    opcode("BRA     ", lb);
}

/*----------------------------------------------------------------*/
/* FOR                                                            */
/*----------------------------------------------------------------*/
void cAsmFile::for1(string lb1)
{
    label(lb1);
}

void cAsmFile::for2(string lb2, string lb3, string lb4)
{
    opcode("CMP     ", "R1  0");
    opcode("BEQ     ", lb4);
    opcode("BRA     ", lb3);
    label(lb2);
}

void cAsmFile::for3(string lb1, string lb3)
{
    opcode("BRA     ", lb1);
    label(lb3);
}

void cAsmFile::for4(string lb2, string lb4)
{
    opcode("BRA     ", lb2);
    label(lb4);
}

/*----------------------------------------------------------------*/
/*  ALLOCATE LOCAL AREA IN STACK                                  */
/*----------------------------------------------------------------*/
void cAsmFile::enter(int n)
{
    char v[BUF_SIZE];
    sprintf(v, "%d", n);
    opcode("ENTER   ", v);
}

void cAsmFile::leave()
{
    opcode("LEAVE   ", "");
}

/*----------------------------------------------------------------*/
/*  LOCAL VARIABLE REFFERENCE                                     */
/*----------------------------------------------------------------*/
void cAsmFile::local_ref(int val, int nest)
{
    char v[BUF_SIZE];

    if (nest >= 0) {
        opcode("LOADBP  ", "R5  0#");
        while (nest > 0) {
            opcode("LOAD    ", "R6  R5");
            opcode("XLOAD   ", "R5  R6");
            nest--;
        }
        sprintf(v, "R5  -%d", val);
        opcode("ADD     ", v);
        opcode("XLOAD   ", "R4  R5");
    } else {
        sprintf(v, "R4  -%d#", val);
        opcode("LOADBP  ", v);
    }
}

/*----------------------------------------------------------------*/
/*  ASSIGN LOCAL VARIABLE                                         */
/*----------------------------------------------------------------*/
void cAsmFile::assign_local(int val, int nest)
{
    char v[BUF_SIZE];

    if (nest > 0) {
        opcode("LOADBP  ", "R5  0#");
        while (nest > 0) {
            opcode("LOAD    ", "R6  R5");
            opcode("XLOAD   ", "R5  R6");
            nest--;
        }
        sprintf(v, "R5  -%d#", val);
        opcode("ADD     ", v);
        opcode("XSTORE  ", "R1  R5");
    } else {
        sprintf(v, "R1  -%d#", val);
        opcode("STORBP  ", v);
    }
}

/*----------------------------------------------------------------*/
/*  SET FORMAL PARAMETER                                          */
/*----------------------------------------------------------------*/
void cAsmFile::set_param()
{
    opcode("PUSH    ","R1");
}

/*----------------------------------------------------------------*/
/*  METHPD CALL                                                   */
/*----------------------------------------------------------------*/
void cAsmFile::method_call(string cname, string mname)
{
    opcode("CALL    ", cname + "_" + mname + ":");
}

void cAsmFile::method_return(int param_no)
{
    for (int i=0; i<param_no; i++) opcode("POP     ","R6 ; discard");
}

/*----------------------------------------------------------------*/
/*  FUNCTION PARAMETER                                            */
/*----------------------------------------------------------------*/
void cAsmFile::func_param1()
{
    opcode("PUSH    ","R1");
}

void cAsmFile::func_param2()
{
    opcode("LOAD    ","R2  R1");
    opcode("POP     ","R1");
}

/*----------------------------------------------------------------*/
/*  RX - RB REGISTER                                              */
/*----------------------------------------------------------------*/
void cAsmFile::assign_register(int sym)
{
    switch(sym) {
        case sym_RQ : opcode("LOAD    ","RQ  R1"); break;
        case sym_RL : opcode("LOAD    ","RL  R1"); break;
        case sym_RX : opcode("LOAD    ","RX  R1"); break;
        case sym_RY : opcode("LOAD    ","RY  R1"); break;
        case sym_RZ : opcode("LOAD    ","RZ  R1"); break;
        case sym_RH : opcode("LOAD    ","RH  R1"); break;
        case sym_RP : opcode("LOAD    ","RP  R1"); break;
        case sym_RB : opcode("LOAD    ","RB  R1"); break;
    }
}

void cAsmFile::ref_register(int sym)
{
    switch(sym) {
        case sym_RQ : opcode("LOAD    ","R4  RQ"); break;
        case sym_RL : opcode("LOAD    ","R4  RL"); break;
        case sym_RX : opcode("LOAD    ","R4  RX"); break;
        case sym_RY : opcode("LOAD    ","R4  RY"); break;
        case sym_RZ : opcode("LOAD    ","R4  RZ"); break;
        case sym_RH : opcode("LOAD    ","R4  RH"); break;
        case sym_RP : opcode("LOAD    ","R4  RP"); break;
        case sym_RB : opcode("LOAD    ","R4  RB"); break;
    }
}

/*----------------------------------------------------------------*/
/*                                                                */
/*----------------------------------------------------------------*/

