/*-------------------------------------------------------------------------*/
/*  J3W   3D Animation Kit  (compiler)                                     */
/*  j3c_pars.cpp   08/06/2012                                              */
/*  Copyright (C) 1998 - 2012   Jun Mizutani <mizutani.jun@nifty.ne.jp>    */
/*                      All rights reserved.                               */
/*                                                                         */
/*   This file is part of the J3W 3D Animation Kit, and is covered under   */
/*  the terms of the GNU General Public License, version 2. This file has  */
/*  NO WARRANTY. See file COPYING for copyright details.                   */
/*                                                                         */
/*-------------------------------------------------------------------------*/

#include <iomanip>
#include <stdio.h>
#include "j3c_pars.h"
#include "j3c_word.h"
#include "j3c_func.h"
#include "j3c_asm.h"
#include "j3c_misc.h"

using namespace std;

cParse::cParse()
{
    scan = new cScanFiles(10);
    SymTable = new cSymRoot(2000,100);
    ClassTable = new cClassTable(200);
    ParamFormat = new cParamFormat(10);
    break_label = "";
    continue_label = "";
    const_str = "";
}

cParse::~cParse()
{
    if (!std_out) os.close();
    delete scan;
    delete code;
    delete SymTable;
}

void cParse::error(string str)
{
    char buff[LINEMAX];
    cerr <<  '\n' << " File: " << scan->GetCurrentFileName() << '\n';
    cerr <<  " Error: " << str << '\n';
    cerr << setw(4) << scan->GetLineCount()<< ":" << scan->GetCurrentLine() <<'\n';
    cerr << setw(scan->GetColumn()+5) <<"^\n";
    scan ->DecodeSymbol(buff, sym);
    cerr << "     Symbol :<" << buff << ">\n";
    err = 1;

#ifdef DEBUG_SCAN
    SymTable->PrintSymbols();
#endif
    delete scan;
    delete code;
    delete SymTable;
    exit(1);
}

void cParse::warning(const char* str)
{
    cerr << " Warning: " << str << '\n';
    cerr << scan->GetLineCount()<< " :" << scan->GetCurrentLine() <<'\n';
}

void cParse::debug(int no, int sym)
{
    scan->DecodeSymbol(buffer, sym);
    cout << "@"<<no<<":"<<buffer << "\n";
}

int cParse::CheckIdent(int s)
{
    char *class_name;
    int  k=0;
    int  n=0;
    int  v=0;

    class_name = SymTable->SearchP(scan->Word(), k, v, n);
    if (class_name) return k == s;
    return 0;
}

void cParse::check_class()
{
    int found;
    string class_name;

    found = SymTable->Search("INIT", sym, value, nest);
    if (!found) {
        class_name = SymTable->SearchP("INIT", sym, value, nest);
        if (class_name == "") error("INIT Method required.");
        else {
            code->label(label.get_init_label());
            code->jump(class_name + "_INIT:");
        }
    }

    int run_found = SymTable->Search("RUN", sym, value, nest);
    if (!run_found) {
        class_name = SymTable->SearchP("RUN", sym, value, nest);
        if (class_name == "") error("RUN Method required.");
        else {
            code->label(label.get_run_label());
            code->jump(class_name + "_RUN:");
        }
    }

    found = SymTable->Search("EVENT", sym, value, nest);
    if (!run_found && found) error("RUN Method required in this class");
    if (!found) {
        class_name = SymTable->SearchP("EVENT", sym, value, nest);
        if (class_name == "") error("EVENT Method required.");
        else {
            code->label(label.get_event_label());
            code->jump(class_name + "_EVENT:");
        }
    }

}

int cParse::parse(char *FileName, char *OutputFile)
{
     int status = scan->OpenFile(FileName);
     if (strlen(OutputFile) == 0) {
         std_out = 1;
         code = new cAsmFile(cout);
     } else {
         std_out = 0;
         setlocale(LC_ALL, "");
         os.open(OutputFile);
         code = new cAsmFile(os);
     }
     if (status == 0) return 1;
     sym = scan->getsym();
     code->label(";+----------------------------------------------+");
     code->label(";|      This file was compiled by j3c.          |");
     code->label(";+----------------------------------------------+");
     code->eol();
     code->jump("main:");
     while((sym == sym_CLASS)||(sym == sym_LIBRARY)||(sym == sym_IMPORT)){
         switch(sym) {
             case sym_CLASS :
                  SymTable->New(scan->Word());
                  def_class();
                  check_class();
                  code->eol();
                  break;
             case sym_LIBRARY :
                  SymTable->New(scan->Word());
                  def_library();
                  code->eol();
                  break;
             case sym_IMPORT :
                  def_import();
                  break;
         }
         sym = scan->getsym();
     }
     if (sym != sym_EOF) error("class , library, or import required.");
     ClassTable->Check();
     return err;
}

void cParse::def_import()
{
    int already_exist = 0;
    char ImportFile[FILENAME_LEN];
    char fname[FILENAME_LEN];

    sym = scan->getsym();
    if (sym == sym_STRING) {
        ExtractFileName(scan->Word(), fname);
        already_exist = ClassTable->CheckFileIdent(fname);
        ClassTable->Register(fname, 0);
        if (!already_exist) strcpy(ImportFile, scan->Word());
    } else error("file name required.(import)");
    sym = scan->getsym();
    if (sym != sym_SEMI) error(" ; required.");
    if (!already_exist) scan->OpenFile(ImportFile);
}

void cParse::def_library()
{
    ParsingLibrary = 1;
    sym = scan->getsym();
    if (sym == sym_IDENT) {
        ClassTable->Register(scan->Word(),2);
        SymTable->Register(scan->Word(), sym_LIBNAME, 0);
        label.set_cname(scan->Word());
        SymTable->set_classname(scan->Word());
        sym = scan->getsym();
    } else error("library name required.");
    if  (sym == sym_LBRACE) sym = scan->getsym();
    else error("{ required. (library)");
    while (sym != sym_RBRACE) {
        if (sym == sym_FINAL) {
            def_var_field();
        } else {
            def_method();
        }
    }
    ParsingLibrary = 0;
}

void cParse::def_class()
{
    ParsingLibrary = 0;
    sym = scan->getsym();
    if (sym == sym_IDENT) {
         SymTable->Register(scan->Word(), sym_CLASSNAME, 0);
         ClassTable->Register(scan->Word(),1);
         label.set_cname(scan->Word());
         SymTable->set_classname(scan->Word());
         code->begin(scan->Word());
         sym = scan->getsym();
    } else error("class name required.");
    if (sym == sym_EXTENDS) {
        sym = scan->getsym();
        if (sym == sym_IDENT) {

             cHSymTable* p = SymTable->SearchClass(scan->Word());
             if (!p) error("super class not found.");
             SymTable->set_parent(p);
             sym = scan->getsym();
        } else error("super class name required.");
    }
    if  (sym == sym_LBRACE) sym = scan->getsym();
    else error("{ required. (class)");
    while (sym != sym_RBRACE) {
        def_field();
    }
}

void cParse::def_field()
{
    if ((sym == sym_FINAL) || (sym == sym_VOLATILE)) {
        def_var_field();
    } else {
        def_method();
    }
}

void cParse::def_method()
{
    if (CheckIdent(sym_TYPENAME)) {
        sym = scan->getsym();
        if (sym == sym_IDENT) {
            int entry = SymTable->Register(scan->Word(), sym_METHODNAME, 0);
            label.set_mname(scan->Word());

            int special = 0;
            if (strcmp(scan->Word(),"RUN") == 0) special = 1;
            else if (strcmp(scan->Word(),"INIT") == 0) special = 2;
            else if (strcmp(scan->Word(),"EVENT") == 0) special = 3;
            switch (special) {
              case 0 : code->method_begin(label.get_method_label()); break;
              case 1 : code->run_begin(label.get_cname()); break;
              case 2 : code->init_begin(label.get_cname()); break;
              case 3 : code->event_begin(label.get_cname()); break;
            }
            sym = scan->getsym();
            if (sym == sym_LPAREN) {
                SymTable->Enter();

                sym = scan->getsym();
                if (CheckIdent(sym_TYPENAME)) {
                    int n = param_list();
                    SymTable->SetValue(entry, n);
                }
                if (sym == sym_RPAREN) {
                    sym = scan->getsym();
                    block(1);
                } else error(" ) required.(def Method)");
                SymTable->Leave();
                switch (special) {
                  case 0 : code->method_end(); break;
                  case 1 : code->run_end(label.get_cname()); break;
                  case 2 : code->init_end(); break;
                  case 3 : code->event_end(label.get_cname()); break;
                }
            } else error (" ( required.(def Method)");
        } else {
            error("Method Name required.");
        }
    } else {
        error("Type Name required.(def Method)");
    }
}

void cParse::def_var_field()
{
    int constant = 0;
    int tbl_pos =0;

    if (sym == sym_FINAL) constant = 1;

    sym = scan->getsym();
    if (CheckIdent(sym_TYPENAME)) {
        sym = scan->getsym();
        do {
            if (sym == sym_IDENT) {
                char var_name[AL];
                strcpy(var_name, scan->Word());
                sym = scan->getsym();
                if (constant)
                    tbl_pos = SymTable->Register(var_name, sym_CONSTANT, 0);
                else {
                    int var_sym = sym_VARNAME;
                    if (sym == sym_LBRACK) var_sym = sym_ARRAYNAME;
                    tbl_pos = SymTable->Register(var_name, var_sym, 0);
                }
                def_var(tbl_pos, constant);
            } else error(" identifier required.");
            if (sym == sym_COMMA) sym = scan->getsym();
            else if (sym != sym_SEMI) error(" ; required.");
        } while (sym != sym_SEMI);
        sym = scan->getsym();
    } else error(" type name required.(def_var_field)");
}

void cParse::def_var(int tbl_pos, int constant)
{
    if (sym == sym_LBRACK) {
        sym = scan->getsym();
        if (constant) {
            if (SymTable->Reserve_Cons_Area(const_exp() - 1)<0)
                error("SymbolTable overflow.");
        } else SymTable->Reserve_Var_Area(const_exp() - 1);
        if (sym == sym_RBRACK) sym = scan->getsym();
        else error("] required. Array Initialize");
    }
    if (sym == sym_EQUAL) {
        if (!constant) error("cannot initialize volatile variable.");
        sym = scan->getsym();
        init_var(tbl_pos);
    }
}

void cParse::init_var(int tbl_pos)
{
    if (sym == sym_LBRACE) {
        sym = scan->getsym();
        const_exp_list(tbl_pos);
        if (sym == sym_RBRACE) sym = scan->getsym();
        else error("} required. Array Initialize");
    } else {
        SymTable->SetValue(tbl_pos, const_exp());
    }
}

void cParse::const_exp_list(int tbl_pos)
{
    int n = const_exp();
    int i = 0;
    SymTable->SetValue(tbl_pos, n);
    i++;
    while (sym == sym_COMMA) {
        sym = scan->getsym();
        n = const_exp();
        SymTable->SetValue(tbl_pos + i, n);
        i++;
    }
}

int cParse::param_list()
{
    int num = 0;
    int pos = 0;
    pos = def_param(num++);
    while (sym == sym_COMMA) {
        sym = scan->getsym();
        def_param(num++);
    }
    for (int i=0; i<num; i++) SymTable->SetValue(pos+i, (num - i + 1));
    return num;
}

int cParse::def_param(int num)
{
    int pos = 0;
    if (CheckIdent(sym_TYPENAME)) {

        sym = scan->getsym();
        if (sym == sym_IDENT) {

            pos = SymTable->Register(scan->Word(), sym_PARAMNAME, num);
            sym = scan->getsym();
        } else error("variable required.(param_list)");
    } else error("type required.(param_list)");
    return pos;
}

void cParse::block(int meth)
{
    int n_local = 0;
    if (sym == sym_LBRACE) {
        sym = scan->getsym();
        if ((meth == 1)||(sym == sym_FINAL)||(sym == sym_VOLATILE))
            SymTable->Enter();
        while ((sym == sym_FINAL)||(sym == sym_VOLATILE)) {
           n_local++;
           local_var_decl(n_local);
        }

        if ((meth == 1) || (n_local > 0)) {
            code->enter(n_local);
        }
        while (sym != sym_RBRACE) {
            statement_list();
        }
        sym = scan->getsym();
        if ((meth == 1) || (n_local > 0)) {
            SymTable->Leave();
            code->leave();
        }
    } else error("} required.(block)");
}

void cParse::statement_list()
{
    statement();
    while (sym == sym_SEMI) {
        sym = scan->getsym();
        if (sym != sym_RBRACE) statement();
    }
}

void cParse::local_var_decl(int &n)
{
    int constant = 0;
    int tbl_pos = 0;
    if (sym == sym_FINAL) constant = 1;

    sym = scan->getsym();
    if (CheckIdent(sym_TYPENAME)) {
        sym = scan->getsym();
        do {
            if (sym == sym_IDENT) {
                if (constant)
                    tbl_pos = SymTable->Register(scan->Word(), sym_CONSTANT, 0);
                else
                    tbl_pos = SymTable->Register(scan->Word(), sym_LOCALNAME, n);
                sym = scan->getsym();
                if (sym == sym_LBRACK) error("array not allowed.");
                if (sym == sym_EQUAL) {
                    sym = scan->getsym();
                    def_local_var(tbl_pos, constant);
                }
            } else error(" identifier required.");
            if (sym == sym_COMMA) sym = scan->getsym();
            else if (sym != sym_SEMI) error(" ; required.");
            n++;
        } while (sym != sym_SEMI);
        n--;
        sym = scan->getsym();
    } else error(" type name required.(def_var_field)");
}

void cParse::def_local_var(int tbl_pos, int type) {
    if (type == 1)
        SymTable->SetValue(tbl_pos, const_exp());
    else {

        error("initialization of local volatile variable not allowed.");
    }
}

void cParse::statement()
{
    switch(sym) {
      case sym_SEMI:
           stat_empty();
           break;
      case sym_SWITCH:
           stat_switch();
           break;
      case sym_WHILE:
           stat_while();
           break;
      case sym_DO:
           stat_do();
           break;
      case sym_FOR:
           stat_for();
           break;
      case sym_IF:
           stat_if();
           break;
      case sym_BREAK:
           stat_break();
           break;
      case sym_CONTINUE:
           stat_continue();
           break;
      case sym_RETURN:
           stat_return();
           break;
      case sym_IDENT:
           if (CheckIdent(sym_VARNAME)) assign_stat();
           else if (CheckIdent(sym_ARRAYNAME)) assign_stat();
           else if (CheckIdent(sym_PARAMNAME)) assign_stat();
           else if (CheckIdent(sym_LOCALNAME)) assign_stat();
           else if (CheckIdent(sym_METHODNAME)) method_call();
           else if (ClassTable->CheckLibraryIdent(scan->Word()))
                   lib_call();
           else error("Unknown Identifier.(statement)");
           break;
      case sym_RQ:
      case sym_RL:
      case sym_RX:
      case sym_RY:
      case sym_RZ:
      case sym_RH:
      case sym_RP:
      case sym_RB:
           assign_register(sym);
           break;
      case sym_LBRACE:
           block(0);
           break;

      case sym_ASM:
           stat_asm();
           break;

      default:
           if (sym >= NOOP) func_call(sym, 0);
           else error("Unknown statement.");
           break;
    }
}

void cParse::stat_empty()
{
    sym = scan->getsym();
}

void cParse::assign_stat()
{
    int sym1;
    int  val;
    int  nest;

    SymTable->SearchP(scan->Word(), sym1, val, nest);
    sym = scan->getsym();
    if (sym == sym_LBRACK) {
        if (sym1 != sym_ARRAYNAME) error("Array variable required. (assign)");
        sym = scan->getsym();
        expression();
        code->assign_array1();
        if (sym == sym_RBRACK) {
            sym = scan->getsym();
        } else {
            error("] required.(array)");
        }
        if (sym == sym_EQUAL) {
            sym = scan->getsym();
            expression();

            code->assign_array2(val);
        } else {
            error("= required.(assign array)");
        }
    } else if (sym == sym_EQUAL) {
        sym = scan->getsym();
        expression();

        if (sym1 == sym_VARNAME) code->assign(val);

        else if (sym1 == sym_PARAMNAME) code->assign_param(val);

        else if (sym1 == sym_LOCALNAME)
            code->assign_local(val, SymTable->GetNest() - 1 - nest);
        else error("Internal error(assignment) ");
    } else {
        error(" = required. (assign variable)");
    }
}

void cParse::assign_register(int sym_reg)
{
    sym = scan->getsym();
    if (sym == sym_EQUAL) {
        sym = scan->getsym();
        expression();
        code->assign_register(sym_reg);
    } else {
        error(" = required. (assign register)");
    }
}

void cParse::method_call()
{
    int  val;

    string meth_name = scan->Word();
    string class_name = SymTable->SearchP(scan->Word(), kind, val, nest);
    sym = scan->getsym();
    if (sym == sym_LPAREN) {
        sym = scan->getsym();
        if (sym != sym_RPAREN) {
           if (formal_param_list() != val)
               error("no. of parameter is different from definition.");
           if (sym == sym_RPAREN) sym = scan->getsym();
           else error (" ) required. (method call)");
        } else {
           sym = scan->getsym();
        }
    } else error(" ( required. (method call)");
    code->method_call(class_name, meth_name);
    code->method_return(val);
}

void cParse::lib_call()
{
    int  param_count;
    char lib_name[AL];
    char meth_name[AL];
    int tblpos;
    cHSymTable* tblp;

    strcpy(lib_name, scan->Word());
    sym = scan->getsym();
    if (sym == sym_DOT) sym = scan->getsym();
    else error(". required. (library call)");
    strcpy(meth_name, scan->Word());
    int sym_type;
    tblp = SymTable->SearchLibrary(lib_name, meth_name, sym_type,
                                   param_count, tblpos);
    if (tblp == 0) error("Unknown library name");
    sym = scan->getsym();
    if (sym == sym_LPAREN) {
        sym = scan->getsym();
        if (sym != sym_RPAREN) {
           if (formal_param_list() != param_count)
               error("no. of parameter is different from definition.");
           if (sym == sym_RPAREN) sym = scan->getsym();
           else error (" ) required. (library call)");
        } else {
           sym = scan->getsym();
        }
        code->method_call(lib_name, meth_name);
        code->method_return(param_count);
    } else if (sym_type == sym_CONSTANT) {
        lib_const(tblp, tblpos);
    } else error(" ( required. (library call)");
}

void cParse::lib_const(cHSymTable* tblp, int tblpos)
{
    int  val;
    int k, n;

    if (sym == sym_LBRACK) {

        sym = scan->getsym();
        tblp->GetValue(tblpos + const_exp(), k, val, n);
        if (sym == sym_RBRACK) {
            code->number(val);
            sym = scan->getsym();
        } else {
            error("] required.(array)");
        }
    } else {
        tblp->GetValue(tblpos, k, val, n);
        code->number(val);
    }
}

int cParse::formal_param_list()
{
    int no_param = 1;
    expression();
    code->set_param();
    while (sym == sym_COMMA) {
        sym = scan->getsym();
        expression();
        code->set_param();
        no_param++;
    }
    return no_param;
}

void cParse::func_call(int op, int f_count)
{
    int RegCount;
    if (ParamFormat->get_format(op) == 0 )
        error("this function is not supported.");
    sym = scan->getsym();
    if (sym == sym_LPAREN) {
        sym = scan->getsym();
        RegCount = func_param_list(f_count);
        if (sym == sym_RPAREN) sym = scan->getsym();
        else error (" ) required. (lib call)");
    } else error(" ( required. (lib call)");
    code->opcode(scan->GetWordName(op), ParamFormat->get_operand());
    ParamFormat->end_format();
}

int cParse::func_param_list(int f_count)
{
    int pf0 = 0;
    int hide = ParamFormat->hide_register();
    int RegCount = 0;
    if (hide) {
        if (f_count == 1) ParamFormat->add_format(" R4 ");
        else ParamFormat->add_format(" R1 ");
        RegCount++;
    }
    do {
        pf0 = ParamFormat->get_field();
        switch(pf0) {
            case 1 : RegCount++;
                     if (RegCount==1) ParamFormat->add_format(" R1 ");
                     if (RegCount==2) {
                         if (hide) ParamFormat->add_format(" R1 ");
                         else ParamFormat->add_format(" R2 ");
                     }
                     if ((RegCount==2) && (!hide)) code->func_param1();
                     expression();
                     if ((RegCount==2) && (!hide)) code->func_param2();
                     break;
            case 2 :
                     if (sym == sym_STRING)
                         ParamFormat->add_str(scan->Word());
                     else error("string required.(function)");
                     sym = scan->getsym();
                     break;
            case 3 : int n = const_exp();
                     if (const_str != "") ParamFormat->add_format(const_str + ": ");
                     else ParamFormat->add_value(n);
                     break;
        }
        if (sym == sym_COMMA) sym = scan->getsym();
    } while ((pf0 > 0) && (sym!=sym_RPAREN));
    return RegCount;
}

void cParse::stat_asm()
{
    sym = scan->getsym();
    if (sym == sym_LPAREN) {
        sym = scan->getsym();
        if (sym == sym_STRING) {
            code->opcode(scan->Word()," ; asm");
            sym = scan->getsym();
        }
        if (sym == sym_RPAREN) sym = scan->getsym();
        else error(") required. asm");
    } else {
        error("( required. asm");
    }
}

void cParse::stat_switch()
{
    string lb1 = break_label;
    break_label = label.get_label();
    sym = scan->getsym();
    if (sym == sym_LPAREN) sym = scan->getsym();
    else error(" ( required. switch");
    expression();
    if (sym == sym_RPAREN) sym = scan->getsym();
    else error(" ) required. switch");
    string lb2 = label.get_label();
    code->switch1();
    if (sym == sym_LBRACE) {
        sym = scan->getsym();
        SymTable->Enter();
    } else error(" { required. switch");
    string fall_lb = label.get_label();

    while (sym != sym_RBRACE) {
        lb2 = switch_block(lb2, fall_lb);
        if ((sym!=sym_RBRACE) && (sym!=sym_CASE) && (sym!=sym_DEFAULT)) {
            error(" ; required. switch");
        }
        fall_lb = label.get_label();
        code->switch5(fall_lb);
    }
    if (sym == sym_RBRACE) {
        code->switch4(fall_lb);
        code->switch4(break_label);
        code->switch3(lb2);
        sym = scan->getsym();
        SymTable->Leave();
    } else error(" } required. switch");

    break_label = lb1;
}

string cParse::switch_block(string lb1, string fall_lb)
{
    while ((sym == sym_CASE) || (sym == sym_DEFAULT)) {
        if (sym == sym_CASE) {
              sym = scan->getsym();
              string lb2 = label.get_label();
              code->switch2(lb1, lb2, fall_lb, const_exp());
              fall_lb = "";
              lb1 = lb2;
              if (sym == sym_COLON) sym = scan->getsym();
              else error(" : required. case");
        } else if (sym == sym_DEFAULT) {
              string lb2 = label.get_label();
              code->switch4(lb1);
              code->switch4(fall_lb);
              fall_lb = "";
              lb1 = lb2;
              sym = scan->getsym();
              if (sym == sym_COLON) sym = scan->getsym();
              else error(" : required. default");
        } else error("case or default required. ");
    }
    statement();
    while (sym == sym_SEMI) {
        sym = scan->getsym();
        if ((sym!=sym_RBRACE) && (sym!=sym_CASE) && (sym!=sym_DEFAULT))
            statement();
    }
    return lb1;
}

void cParse::stat_while()
{
    string lb1 = continue_label;
    continue_label = label.get_label();
    string lb2 = break_label;
    break_label = label.get_label();
    code->while1(continue_label);
    sym = scan->getsym();
    if (sym == sym_LPAREN) sym = scan->getsym();
    else error(" ( required. while");
    expression();
    code->while2(break_label);
    if (sym == sym_RPAREN) sym = scan->getsym();
    else error(" ) required. while");
    statement();
    code->while3(continue_label, break_label);
    continue_label = lb1;
    break_label = lb2;
}

void cParse::stat_do()
{
    sym = scan->getsym();
    string lb1 = label.get_label();
    code->do1(lb1);
    statement();
    if (sym == sym_WHILE) sym = scan->getsym();
    else error(" while required. do");
    if (sym == sym_LPAREN) sym = scan->getsym();
    else error(" ( required. do");
    expression();
    code->do2(lb1, label.get_label());
    if (sym == sym_RPAREN) sym = scan->getsym();
    else error(" ) required. do");
    statement();
}

void cParse::stat_for()
{
    string lb1 = label.get_label();
    string lb2 = label.get_label();
    string lb3 = label.get_label();
    string lb4 = label.get_label();
    sym = scan->getsym();
    if (sym == sym_LPAREN) sym = scan->getsym();
    else error(" ( required. for");

    if (sym != sym_SEMI) {
        statement();
        if (sym == sym_SEMI) sym = scan->getsym();
        else error("; required. (for)");
    } else sym = scan->getsym();
    code->for1(lb1);

    if (sym != sym_SEMI) {
        expression();
        if (sym == sym_SEMI) sym = scan->getsym();
        else error("; required. (for)");
    } else sym = scan->getsym();
    code->for2(lb2,lb3,lb4);

    if (sym != sym_RPAREN) {
        statement();
        if (sym == sym_RPAREN) sym = scan->getsym();
        else error(") required. (for)");
    } else sym = scan->getsym();
    code->for3(lb1, lb3);
    statement();
    code->for4(lb2, lb4);
}

void cParse::stat_if()
{
    sym = scan->getsym();
    if (sym == sym_LPAREN) sym = scan->getsym();
    else error(" ( required. if");
    expression();
    if (sym == sym_RPAREN) sym = scan->getsym();
    else error(" ) required. if");
    string lb1 = label.get_label();
    code->if1(lb1);
    statement();
    string lb2 = label.get_label();
    code->if2(lb1, lb2);
    if (sym == sym_ELSE) {
        sym = scan->getsym();
        statement();
    }
    code->if3(lb2);
}

void cParse::stat_return()
{
    sym = scan->getsym();
    if ((sym != sym_SEMI) && (sym != sym_RBRACE)) {
        expression();
    }
    code->return0();
}

void cParse::stat_break()
{
    code->break0(break_label);
    sym = scan->getsym();
}

void cParse::stat_continue()
{
    code->continue0(continue_label);
    sym = scan->getsym();
}

void cParse::expression()
{
    code->enter_exp();
    code->opt_begin();
    simple_expression();
    code->exp1();
    if (  (sym == sym_2EQL) || (sym == sym_GTR) || (sym == sym_LESS)
        || (sym == sym_GTREQL) || (sym == sym_LESSEQL)
        || (sym == sym_EXEQL) ) {
        int op = sym;
        sym = scan->getsym();
        simple_expression();
        code->exp2(op, label.get_label(), label.get_label());
    }
    code->exit_exp();
}

void cParse::simple_expression()
{
    term();
    if ((sym != sym_PLUS) && (sym != sym_MINUS) && (sym != sym_BAR))
        code->opt_exp();
    else code->simple_exp1();
    while((sym == sym_PLUS)||(sym == sym_MINUS)||(sym == sym_BAR)) {
        int op = sym;
        sym = scan->getsym();
        term();
        code->simple_exp2(op);
    }
}

void cParse::term()
{
    factor();
    if ((sym != sym_MUL) && (sym != sym_SLASH) && (sym != sym_AMP))
        code->opt_exp();
    else code->term1();
    while((sym == sym_MUL)||(sym == sym_SLASH)||(sym == sym_AMP)) {
        int op = sym;
        sym = scan->getsym();
        factor();
        code->term2(op);
    }
}

void cParse::factor()
{
    code->count_factor();
    switch(sym) {
      case sym_LPAREN:
           sym = scan->getsym();
           code->paren1();
           expression();
           code->paren2();
           if (sym == sym_RPAREN) sym = scan->getsym();
           else error(") required. factor");
           break;
      case sym_MINUS:
           sym = scan->getsym();
           factor();
           code->minus();
           break;
      case sym_EXCL:
           sym = scan->getsym();
           factor();
           code->not0();
           break;
      case sym_NUMBER:
           code->number(scan->Num());
           sym = scan->getsym();
           break;

      case sym_IDENT:
           if (CheckIdent(sym_VARNAME)) variable();
           else if (CheckIdent(sym_PARAMNAME)) variable();
           else if (CheckIdent(sym_LOCALNAME)) variable();
           else if (CheckIdent(sym_ARRAYNAME)) array();
           else if (CheckIdent(sym_CONSTANT)) constant();
           else if (CheckIdent(sym_METHODNAME)) {
               int cnt = code->get_factor_count();
               if (cnt != 1) code->paren1();
               method_call();
               if (cnt != 1) code->paren2();
           } else if (ClassTable->CheckLibraryIdent(scan->Word()))
               lib_call();
           else error("Unknown Identifier.(factor)");
           break;

      case sym_RQ:
      case sym_RL:
      case sym_RX:
      case sym_RY:
      case sym_RZ:
      case sym_RH:
      case sym_RP:
      case sym_RB:
           ref_register(sym);
           sym = scan->getsym();
           break;

      default:
           if (sym >= NOOP) {
               int cnt = code->get_factor_count();
               if (cnt != 1) code->paren1();
               func_call(sym, code->get_factor_count());
               if (cnt != 1) code->paren2();
           } else error("Unknown Symbol(factor)");
           break;
    }
}

void cParse::ref_register(int sym)
{
    code->ref_register(sym);
}

void cParse::array()
{
    int  val;
    int sym1 = 0;

    SymTable->SearchP(scan->Word(), sym1, val, nest);
    sym = scan->getsym();
    if (sym == sym_LBRACK) {

        sym = scan->getsym();
        code->array_ref1();
        expression();
        if (sym == sym_RBRACK) {
            code->array_ref2(val);
            sym = scan->getsym();
        } else {
            error("] required.(array)");
        }
    } else code->array_ref(val);
}

void cParse::variable()
{
    int  val;
    int sym1 = 0;

    SymTable->SearchP(scan->Word(), sym1, val, nest);
    sym = scan->getsym();
    if (sym == sym_LBRACK) error("not array variable");
    else {
        if (sym1 == sym_VARNAME) code->var_ref(val);
        else if (sym1 == sym_PARAMNAME) code->param_ref(val);

        else if (sym1 == sym_LOCALNAME)
            code->local_ref(val, SymTable->GetNest() - nest - 1);
        else error(" internal error(variable reference) ");

    }
}

void cParse::constant()
{
    int  val;
    int k, n;
    int tblpos;

    cHSymTable* tbl = SymTable->SearchT(scan->Word(), tblpos);
    if (!tbl) error("Unknown constant name.");
    sym = scan->getsym();
    if (sym == sym_LBRACK) {

        sym = scan->getsym();
        tbl->GetValue(tblpos + const_exp(), k, val, n);
        if (sym == sym_RBRACK) {
            code->number(val);
            sym = scan->getsym();
        } else {
            error("] required.(array)");
        }
    } else {
        tbl->GetValue(tblpos, k, val, n);
        code->number(val);
    }
}

int cParse::const_exp()
{
    const_str = "";
    if (sym == sym_IDENT) {
        if (ClassTable->CheckClassIdent(scan->Word())) {
            const_str = scan->Word();
            sym = scan->getsym();
            return 0;
        }
    }
    int result = const_term();
    while((sym == sym_PLUS)||(sym == sym_MINUS)||(sym == sym_BAR)) {
        if (sym == sym_PLUS) {
            sym = scan->getsym();
            result +=const_term();
        } else if (sym == sym_BAR) {
            sym = scan->getsym();
            result |= const_term();
        } else {
            sym = scan->getsym();
            result -=const_term();
        }
    }
    return result;
}

int cParse::const_term()
{
    int result = const_factor();
    while((sym == sym_MUL)||(sym == sym_SLASH)||(sym == sym_AMP)) {
        if (sym == sym_MUL) {
            sym = scan->getsym();
            result *= const_factor();
        } else if (sym == sym_AMP) {
            sym = scan->getsym();
            result &= const_factor();
        } else {
            sym = scan->getsym();
            result /= const_factor();
        }
    }
    return result;
}

int  cParse::lib_constant()
{
    int  param_count;
    char lib_name[AL];
    char meth_name[AL];
    int tblpos;
    int result = 0;
    cHSymTable* tblp;

    strcpy(lib_name, scan->Word());
    sym = scan->getsym();
    if (sym == sym_DOT) sym = scan->getsym();
    else error(". required. (library constant)");
    strcpy(meth_name, scan->Word());
    int sym_type;
    tblp = SymTable->SearchLibrary(lib_name, meth_name, sym_type,
                                   param_count, tblpos);
    if (tblp == 0) error("Unknown library name");
    sym = scan->getsym();
    if (sym_type == sym_CONSTANT) {
        int  val;
        int k, n;

        if (sym == sym_LBRACK) {
            sym = scan->getsym();
            tblp->GetValue(tblpos + const_exp(), k, val, n);
            if (sym == sym_RBRACK) {
                result = val;
                sym = scan->getsym();
            } else error("] required.(array)");
        } else {
            tblp->GetValue(tblpos, k, val, n);
            result = val;
        }
    } else error(" library constant riquired.)");
    return result;
}

int cParse::const_factor()
{
    int result = 0;
    switch(sym) {
      case sym_LPAREN:
           sym = scan->getsym();
           result = const_exp();
           if (sym == sym_RPAREN) sym = scan->getsym();
           else error(") required. factor");
           break;
      case sym_MINUS:
           sym = scan->getsym();
           result = -const_factor();
           break;
      case sym_NUMBER:
           result = scan->Num();
           sym = scan->getsym();
           break;
      case sym_IDENT:
           if (CheckIdent(sym_CONSTANT)) {
               SymTable->Search(scan->Word(), sym, value, nest);
               sym = scan->getsym();
               result = value;
           } else if (ClassTable->CheckLibraryIdent(scan->Word()))
               result = lib_constant();
           else error("Unknown Identifier.(const_factor)");
           break;
      default:
           error("Unknown Symbol(const_factor)");
           break;
    }
    return result;
}

