/*-------------------------------------------------------------------------*/
/*  J3W   3D Animation Kit  (compiler)                                     */
/*  j3p_scan.cpp   08/06/2012                                              */
/*  Copyright (C) 1998 - 2012 Jun Mizutani <mizutani.jun@nifty.ne.jp>      */
/*                      All rights reserved.                               */
/*                                                                         */
/*   This file is part of the J3W 3D Animation Kit, and is covered under   */
/*  the terms of the GNU General Public License, version 2. This file has  */
/*  NO WARRANTY. See file COPYING for copyright details.                   */
/*                                                                         */
/*-------------------------------------------------------------------------*/

#include <stdio.h>
#include <ctype.h>

#include "j3c_scan.h"
#include "j3c_word.h"
#include "j3c_misc.h"

using namespace std;

cScan::cScan() {
    WdLen = 0;
    strcpy(LineBuf, "");
    strcpy(CurrentLine, "");
    strcpy(SourcePath, "");
    LineLen = 0;
    LinePtr = 0;
}

cScan::~cScan() {
}

void cScan::error(const char* str)
{
      cerr << " Error: " << str << '\n';
      err = 1;
}

int cScan::GetLine() {
    do {
        LineLen = 0;
        if (!fin.eof()) {
            fin.getline(LineBuf,LINEMAX);
        } else {
            fin.close();
            LineLen = 0;
            LinePtr = 0;
            return 0;
        }
        LineCount++;
        strcpy(CurrentLine, LineBuf);
        LineLen = strlen(LineBuf);
    } while (LineLen == 0);

    LinePtr = 0;
    return 1;
}

char cScan::getch()
{
    int f=1;

    if (LinePtr < LineLen) {
      nextch = LineBuf[LinePtr++];

      if ((nextch == '/') && (LinePtr < LineLen)) {
        if (LineBuf[LinePtr] == '/') {
          nextch = ' ';
          LinePtr = LineLen;
        }
      }
    } else if (eof == 0) {
        f = GetLine();
        nextch = ' ';
    }
    if (!f) {
        eof = 1;
        nextch = 0;
        return nextch;
    }
    if ((unsigned char)nextch < ' ') nextch = ' ';
    return nextch;
}

void cScan::GetToken(char delim)
{
    int k = 0;
    getch();
    Wd[0] = 0;
    while ((nextch != delim)&&(nextch != 0)&&(k < LINEMAX)) {
        if (k < LINEMAX) {
            Wd[k++] = nextch;
            Wd[k] = 0;
        }
        getch();
    }
    getch();
}

inline int cScan::CheckAlfaNum()
{
    return    ((nextch >= 'A') && (nextch <= 'Z'))
           || ((nextch >= '0') && (nextch <= '9'))
           || ((nextch >= 'a') && (nextch <= 'z'))
           || (nextch == '_');
}

void cScan::GetIdentifier()
{
    int k = 0;
    Wd[0] = 0;
    while (CheckAlfaNum()) {
        if (k < LINEMAX) {
            Wd[k++] = nextch;
            Wd[k] = 0;
        }
        getch();
    }
    sym = sym_IDENT;
}

void cScan::GetNumber()
{
    sym = sym_NUMBER;

    num = 0;
    int n = 0;

    if (nextch == '0') {
        getch();
        if ((nextch == 'x')||(nextch == 'X')) {
            getch();
            do {
                if (isdigit(nextch))       n = nextch - '0';
                else if (isupper(nextch))  n = nextch - 'A' + 10;
                else if (islower(nextch))  n = nextch - 'a' + 10;
                else error(" Incorrect hexadecimal number format ");
                num = 16 * num + n;
                getch();
            } while (    ((nextch >= '0') && (nextch <= '9'))
                      || ((nextch >= 'A') && (nextch <= 'F'))
                      || ((nextch >= 'a') && (nextch <= 'f')) );
        } else {
            num = 0;
        }
    }

    while ((nextch >= '0') && (nextch <= '9')) {
        num = 10 * num + (nextch - '0');
        getch();
    }
}

int cScan::getsym()
{
    do {
        while (nextch == ' ') getch();

        if ((( nextch >= 'A')&&(nextch <= 'Z'))|| (nextch == '_') ||
            (( nextch >= 'a')&&(nextch <= 'z'))) GetIdentifier();

        else if (( nextch >= '0')&&(nextch <= '9')) GetNumber();

        else
          switch(nextch) {
              case '(' : sym = sym_LPAREN       ;getch(); break;
              case ')' : sym = sym_RPAREN       ;getch(); break;
              case '[' : sym = sym_LBRACK       ;getch(); break;
              case ']' : sym = sym_RBRACK       ;getch(); break;
              case '{' : sym = sym_LBRACE       ;getch(); break;
              case '}' : sym = sym_RBRACE       ;getch(); break;
              case ',' : sym = sym_COMMA        ;getch(); break;
              case ':' : sym = sym_COLON        ;getch(); break;
              case ';' : sym = sym_SEMI         ;getch(); break;
              case '.' : sym = sym_DOT          ;getch(); break;
              case '?' : sym = sym_QUEST        ;getch(); break;
              case '~' : sym = sym_TILDE        ;getch(); break;
              case '"' : GetToken('"');
                         sym = sym_STRING;
                         break;
              case '\'': getch();
                         sym = sym_NUMBER;
                         num = nextch;
                         while (nextch != '\'') getch();
                         getch();
                         break;
              case '/' : getch();
                         if (nextch == '*') {
                             getch();
                             do {
                                 while (nextch != '*') getch();
                                 getch();
                             } while (nextch != '/');
                             getch();
                             sym = sym_NIL;
                         } else if (nextch == '=') {
                             getch();
                             sym = sym_DIVEQL;    /* /=  */
                         } else sym = sym_SLASH;
                         break;
              case '!' : sym = sym_EXCL;
                         getch();
                         if (nextch == '=') {
                             sym = sym_EXEQL;
                             getch();
                         }
                         break;
              case '^' : sym = sym_HAT;
                         getch();
                         if (nextch == '=') {
                             sym = sym_HATEQL;
                             getch();
                         }
                         break;
              case '+' : sym = sym_PLUS;
                         getch();
                         if (nextch == '+') {
                             sym = sym_2PLUS;
                             getch();
                         } else if (nextch == '=') {
                             sym = sym_PLUSEQL;
                             getch();
                         }
                         break;
              case '-' : sym = sym_MINUS;
                         getch();
                         if (nextch == '-') {
                             sym = sym_2MINUS;
                             getch();
                         } else if (nextch == '=') {
                             sym = sym_MINUSEQL;
                             getch();
                         }
                         break;
              case '*' : sym = sym_MUL;
                         getch();
                         if (nextch == '=') {
                             sym = sym_MULEQL;
                             getch();
                         }
                         break;
              case '%' : sym = sym_PERCENT;
                         getch();
                         if (nextch == '=') {
                             sym = sym_PCEQL;
                             getch();
                         }
                         break;
              case '|' : sym = sym_BAR;
                         getch();
                         if (nextch == '|') {
                             sym = sym_2BAR;
                             getch();
                         } else if (nextch == '=') {
                             sym = sym_BAREQL;
                             getch();
                         }
                         break;
              case '&' : sym = sym_AMP;
                         getch();
                         if (nextch == '&') {
                             sym = sym_2AMP;
                             getch();
                         } else if (nextch == '=') {
                             sym = sym_AMPEQL;
                             getch();
                         }
                         break;
              case '=' : sym = sym_EQUAL;
                         getch();
                         if (nextch == '=') {
                             sym = sym_2EQL;
                             getch();
                         }
                         break;
              case '>' : sym = sym_GTR;
                         getch();
                         if (nextch == '>') {
                             sym = sym_RSHIFT;
                             getch();
                             if (nextch == '=') {
                                 getch();
                                 sym = sym_RSHIFTEQL;
                             }
                         } else if (nextch == '=') {
                             sym = sym_GTREQL;
                             getch();
                         }
                         break;
              case '<' : sym = sym_LESS;
                         getch();
                         if (nextch == '<') {
                             sym = sym_LSHIFT;
                             getch();
                             if (nextch == '=') {
                                 getch();
                                 sym = sym_LSHIFTEQL;
                             }
                         } else if (nextch == '=') {
                             sym = sym_LESSEQL;
                             getch();
                         }
                         break;
              case 0   : sym = sym_EOF; break;
              default  : sym = sym_NIL; getch();  break;
          }
    } while (sym == sym_NIL);

    if (sym == sym_IDENT) ResWord.CheckResWord(Wd, sym);
    if (sym == sym_IDENT) ResWord.CheckOpeCode(Wd, sym);
    if (sym == sym_IDENT) ResWord.CheckRegName(Wd, sym);

#ifdef  DEBUG_SCAN
        char buf[LINEMAX];
        DecodeSymbol(buf, sym);
        cout << buf << '\n';
#endif
    return sym;
}

const char* cScan::GetWordName(int sym)
{
    const char *p = ResWord.GetResWordName(sym);
    if (!p) p = ResWord.GetOpecodeName(sym);
    return p;
}

void cScan::DecodeSymbol(char* buffer, int sym)
{
    const char *p = GetWordName(sym);
    if (p) {
      strcpy(buffer,p);
    } else {
        switch(sym) {
            case  sym_EOF          :strcpy(buffer,"[EOF]");break;
            case  sym_IDENT        :strcpy(buffer,Word());break;
            case  sym_NUMBER       :sprintf(buffer,"%d", int(Num()));break;
            case  sym_STRING       :strcpy(buffer,Word());break;
            case  sym_CHARACTER    :strcpy(buffer,"<chr>");break;
            case  sym_CLASSNAME :
            case  sym_LIBNAME   :
            case  sym_VARNAME   :
            case  sym_TYPENAME  :
            case  sym_METHODNAME:
            case  sym_CONSTANT  :  strcpy(buffer,Word());break;

            case  sym_NIL          :strcpy(buffer,"");break;
            case  sym_LPAREN       :strcpy(buffer,"(");break;
            case  sym_RPAREN       :strcpy(buffer,")");break;
            case  sym_LBRACK       :strcpy(buffer,"[");break;
            case  sym_RBRACK       :strcpy(buffer,"]");break;
            case  sym_LBRACE       :strcpy(buffer,"{");break;
            case  sym_RBRACE       :strcpy(buffer,"}");break;
            case  sym_COMMA        :strcpy(buffer,",");break;
            case  sym_COLON        :strcpy(buffer,":");break;
            case  sym_SEMI         :strcpy(buffer,";");break;
            case  sym_DOT          :strcpy(buffer,".");break;
            case  sym_EXCL         :strcpy(buffer,"!");break;
            case  sym_PLUS         :strcpy(buffer,"+");break;
            case  sym_MINUS        :strcpy(buffer,"-");break;
            case  sym_MUL          :strcpy(buffer,"*");break;
            case  sym_SLASH        :strcpy(buffer,"/");break;
            case  sym_PERCENT      :strcpy(buffer,"%");break;
            case  sym_BAR          :strcpy(buffer,"|");break;
            case  sym_AMP          :strcpy(buffer,"&");break;
            case  sym_RSHIFT       :strcpy(buffer,">>");break;
            case  sym_LSHIFT       :strcpy(buffer,"<<");break;
            case  sym_2PLUS        :strcpy(buffer,"++");break;
            case  sym_2MINUS       :strcpy(buffer,"--");break;
            case  sym_EQUAL        :strcpy(buffer,"= ");break;
            case  sym_PCEQL        :strcpy(buffer,"%=");break;
            case  sym_AMPEQL       :strcpy(buffer,"&=");break;
            case  sym_MULEQL       :strcpy(buffer,"*=");break;
            case  sym_PLUSEQL      :strcpy(buffer,"+=");break;
            case  sym_MINUSEQL     :strcpy(buffer,"-=");break;
            case  sym_DIVEQL       :strcpy(buffer,"/=");break;
            case  sym_LSHIFTEQL    :strcpy(buffer,"<<=");break;
            case  sym_RSHIFTEQL    :strcpy(buffer,">>=");break;
            case  sym_HATEQL       :strcpy(buffer,"^=");break;
            case  sym_BAREQL       :strcpy(buffer,"|=");break;
            case  sym_GTR          :strcpy(buffer,">");break;
            case  sym_LESS         :strcpy(buffer,"<");break;
            case  sym_GTREQL       :strcpy(buffer,">=");break;
            case  sym_LESSEQL      :strcpy(buffer,"<=");break;
            case  sym_2AMP         :strcpy(buffer,"&&");break;
            case  sym_2BAR         :strcpy(buffer,"||");break;
            case  sym_2EQL         :strcpy(buffer,"==");break;
            case  sym_EXEQL        :strcpy(buffer,"!=");break;
            case  sym_3GTR         :strcpy(buffer,">>>");break;
            case  sym_3GTREQL      :strcpy(buffer,">>>=");break;
            case  sym_QUEST        :strcpy(buffer,"?");break;
            case  sym_HAT          :strcpy(buffer,"^");break;
            case  sym_TILDE        :strcpy(buffer,"~");break;

            case  sym_RQ           :strcpy(buffer,"RQ"); break;
            case  sym_RL           :strcpy(buffer,"RL"); break;
            case  sym_RX           :strcpy(buffer,"RX"); break;
            case  sym_RY           :strcpy(buffer,"RY"); break;
            case  sym_RZ           :strcpy(buffer,"RZ"); break;
            case  sym_RH           :strcpy(buffer,"RH"); break;
            case  sym_RP           :strcpy(buffer,"RP"); break;
            case  sym_RB           :strcpy(buffer,"RB"); break;
            case  sym_R1           :strcpy(buffer,"R1"); break;
            case  sym_R2           :strcpy(buffer,"R2"); break;
            case  sym_R3           :strcpy(buffer,"R3"); break;
            case  sym_R4           :strcpy(buffer,"R4"); break;
            case  sym_R5           :strcpy(buffer,"R5"); break;
            case  sym_R6           :strcpy(buffer,"R6"); break;
            default : strcpy(buffer, "<unknown>");break;
        }
    }
}

int cScan::OpenFile(char *FileName)
{
    char fullpath[FILENAME_LEN];

    strcpy(fullpath, SourcePath);
    strcat(fullpath, FileName);
    setlocale(LC_ALL, "");
    fin.open(fullpath);
    strcpy(CurrentFileName, fullpath);
    LineCount = 0;
    if (!fin) {
        cerr << " Cannot open file: " << fullpath << '\n';
        return 0;
    } else {
        eof = 0;
        GetLine();
        getch();
        return 1;
    }
}

void cScan::CloseFile()
{
}
