/*-------------------------------------------------------------------------*/
/*  J3W   3D Animation Kit  (compiler)                                     */
/*  j3c_word.cpp   08/06/2012                                              */
/*  Copyright (C) 1998 - 2012 Jun Mizutani <mizutani.jun@nifty.ne.jp>      */
/*                      All rights reserved.                               */
/*                                                                         */
/*   This file is part of the J3W 3D Animation Kit, and is covered under   */
/*  the terms of the GNU General Public License, version 2. This file has  */
/*  NO WARRANTY. See file COPYING for copyright details.                   */
/*                                                                         */
/*-------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>

using namespace std;

#include "j3c_word.h"

struct ResWordType {
    int   sym;
    const char* wn;
};

struct OPType {
    int  op;
    const char* opn;
    const char* wn;
};

ResWordType ResWord[] = {

    {sym_BOOLEAN      ,"boolean"     },
    {sym_BYTE         ,"byte"        },
    {sym_CHAR         ,"char"        },
    {sym_DOUBLE       ,"double"      },
    {sym_FLOAT        ,"float"       },
    {sym_INT          ,"int"         },
    {sym_LONG         ,"long"        },
    {sym_SHORT        ,"short"       },

    {sym_BREAK        ,"break"       },
    {sym_CASE         ,"case"        },
    {sym_CONTINUE     ,"continue"    },
    {sym_DEFAULT      ,"default"     },
    {sym_DO           ,"do"          },
    {sym_ELSE         ,"else"        },
    {sym_FOR          ,"for"         },
    {sym_IF           ,"if"          },

    {sym_RETURN       ,"return"      },
    {sym_SWITCH       ,"switch"      },
    {sym_WHILE        ,"while"       },

    {sym_CLASS        ,"class"       },
    {sym_EXTENDS      ,"extends"     },
    {sym_IMPLEMENTS   ,"implements"  },
    {sym_IMPORT       ,"import"      },
    {sym_INTERFACE    ,"interface"   },
    {sym_PACKAGE      ,"package"     },
    {sym_VOID         ,"void"        },
    {sym_THROWS       ,"throws"      },

    {sym_CATCH        ,"catch"       },
    {sym_FINALLY      ,"finally"     },
    {sym_THROW        ,"throw"       },
    {sym_TRY          ,"try"         },

    {sym_SUPER        ,"super"       },
    {sym_THIS         ,"this"        },

    {sym_ABSTRACT     ,"abstract"    },
    {sym_FINAL        ,"final"       },
    {sym_NATIVE       ,"native"      },
    {sym_PUBLIC       ,"public"      },
    {sym_PRIVATE      ,"private"     },
    {sym_PROTECTED    ,"protected"   },
    {sym_STATIC       ,"static"      },
    {sym_SYNCHRONIZED ,"synchronized"},
    {sym_TRANSIENT    ,"transient"   },
    {sym_VOLATILE     ,"volatile"    },
    {sym_ASM          ,"asm"         },   /* not in JAVA*/
    {sym_LIBRARY      ,"library"     }    /* not in JAVA*/
};

OPType OpeCode[] = {
    { NOOP  ,"NOOP",  "Pause"   },
    { THROW ,"THROW",  "Throw" },
    { INC   ,"INC",  "INC"    },
    { DEC   ,"DEC",  "DEC"   },
    { ADD   ,"ADD",  "ADD"    },
    { SUB   ,"SUB",  "SUB"   },
    { MUL   ,"MUL",  "MUL"    },
    { DIV   ,"DIV",  "DIV"    },
    { SHL   ,"SHL",  "SHL"   },
    { SHR   ,"SHR",  "SHR"    },
    { AND   ,"AND",  "AND"    },
    { OR    ,"OR",  "OR"    },
    { NOT   ,"NOT",  "not"    },
    { CMP   ,"CMP",  "CMP"    },
    { NEG   ,"NEG",  "NEG"   },
    { SQRT  ,"SQRT", "sqrt"   },
    { SIN   ,"SIN",  "sin"    },
    { COS   ,"COS",  "cos"   },
    { ATAN  ,"ATAN", "atan"   },
    { RANDOM,"RANDOM", "Random"},
    { BRA   ,"BRA",  "BRA"    },
    { BEQ   ,"BEQ",  "BEQ"    },
    { BNE   ,"BNE",  "BNE"    },
    { BGR   ,"BGR",  "BGR"    },
    { BGE   ,"BGE",  "BGE"    },
    { BLS   ,"BLS",  "BLS"    },
    { BLE   ,"BLE",  "BLE"    },
    { BMI   ,"BMI",  "BMI"    },
    { BPL   ,"BPL",  "BPL"    },
    { LOOP  ,"LOOP",  "LOOP"   },
    { CALL  ,"CALL",  "CALL"   },
    { RETURN,"RETURN", "RETURN"},
    { PUSH  ,"PUSH",  "PUSH"   },
    { POP   ,"POP",  "POP"    },
    { CLEARA,"CLEARA", "ClearRegisters"},
    { PUSHA ,"PUSHA",  "PushRegisters" },
    { POPA  ,"POPA",  "PopRegisters"   },
    { CLEARG,"CLEARG", "CLEARG"},
    { PUSHG ,"PUSHG",  "PUSHG" },
    { POPG  ,"POPG",  "POPG"   },
    { ENTER ,"ENTER",  "ENTER" },
    { LEAVE ,"LEAVE",  "LEAVE" },
    { CALLTB,"CALLTB", "CALLTB"},
    { LOAD  ,"LOAD",  "LOAD"  },
    { LOADM ,"LOADM", "LOADM" },
    { XLOAD ,"XLOAD", "XLOAD"  },
    { CLOAD ,"CLOAD",  "GetCommon" },
    { LOADBP,"LOADBP", "LOADBP"},
    { PNTUV, "PNTUV", "PointUV"},
    { DEFPUV, "DEFPUV",  "DefPointUV"},
    { TXFILE,"TXFILE", "TextureFile"},
    { TXSET, "TXSET",  "TextureSet"},
    { TXAXIS,"TXAXIS", "TextureAxis"},
    { TXALPH,"TXALPH", "TextureAlpha"},
    { TXSCAL,"TXSCAL", "TextureScale"},
    { TXCHG, "TXCHG",  "TextureChange"},
    { TXMAP, "TXMAP", "TextureMap"},
    { TXBIAS,"TXBIAS", "TextureBias"},
    { STORE ,"STORE",  "STORE" },
    { XSTORE,"XSTORE", "XSTORE"},
    { CSTORE,"CSTORE", "SetCommon"},
    { STORBP,"STORBP", "STORBP"},
    { GENPRC,"GENPRC", "new"},
    { DELPRC,"DELPRC", "delete"},
    { WAIT  ,"WAIT",  "Wait"   },
    { STPALL,"STPALL", "Stop"},
    { SEND  ,"SEND",  "Send"  },
    { RECEIV,"RECEIV", "Receive"},
    { SENDTO,"SENDTO", "SendTo"},
    { CHILD ,"CHILD",  "child" },
    { GRAPHM,"GRAPHM", "GraphMode"},
    { TEXTM ,"TEXTM",  "TextMode" },
    { BCOLOR,"BCOLOR", "BackgroundColor"},
    { ZOOM  ,"ZOOM",  "Zoom"  },
    { WIRE  ,"WIRE",  "Wire"  },
    { NEAR  ,"NEAR",  "Near"  },
    { FAR   ,"FAR",   "Far"  },
    { EMIT  ,"EMIT",  "Emit"  },
    { PARALL,"PARALL", "Parallel"},
    { INKEY ,"INKEY",  "InKey" },
    { RAWKEY,"RAWKEY", "RawKey"},
    { MOUSE ,"MOUSE", "Mouse"  },
    { OUTNUM,"OUTNUM", "Number"},
    { OUTCHR,"OUTCHR", "Char"},
    { OUTSTM,"OUTSTM", "Stringm"},
    { OUTSTR,"OUTSTR", "String"},
    { NOTE  ,"NOTE",  "Note"  },
    { MIDI  ,"MIDI",  "Midi"  },
    { MOVF  ,"MOVF",  "Forward"},
    { MOVB  ,"MOVB",  "Back"  },
    { MOVL  ,"MOVL",  "Left"  },
    { MOVR  ,"MOVR",  "Right"  },
    { MOVU  ,"MOVU",  "Up"  },
    { MOVD  ,"MOVD",  "Down"  },
    { ROTH  ,"ROTH",  "RotHead"  },
    { ROTP  ,"ROTP",  "RotPitch"  },
    { ROTB  ,"ROTB",  "RotBank"  },
    { MOVALL,"MOVALL", "Move0"},
    { MOVALH,"MOVALH", "Move"},
    { SCALE ,"SCALE",  "Scale" },
    { SETGRV,"SETGRV", "SetGravity"},
    { GETGRV,"GETGRV", "GetGravity"},
    { SETACL,"SETACL", "SetAcceleration"},
    { GETACL,"GETACL", "GetAcceleration"},
    { SETVEL,"SETVEL", "SetVelocity"},
    { GETVEL,"GETVEL", "GetVelocity"},
    { SETWVL,"SETWVL", "SetWVelocity"},
    { GETWVL,"GETWVL", "GetWVelocity"},
    { GENOBJ,"GENOBJ", "NewObject"},
    { DEFPNT,"DEFPNT", "DefPoint"},
    { DEFPLN,"DEFPLN", "DefPlane"},
    { DELOBJ,"DELOBJ", "DeleteObject"},
    { SETEYE,"SETEYE", "See"},
    { POINT ,"POINT",  "Point" },
    { SETREF,"SETREF", "Specular"},
    { SETAMB,"SETAMB", "Ambient"},
    { EYEORG,"EYEORG", "EYEORG"},
    { GETVXN,"GETVXN", "CountVertex"},
    { PLANE ,"PLANE",  "Plane" },
    { TRANSP,"TRANSP", "Transpose"},
    { UPDTVP,"UPDTVP", "VertexRelative"},
    { RSETVP,"RSETVP", "VertexAbsolute"},
    { SLIDE ,"SLIDE",  "SetOffset" },
    { HOME  ,"HOME",  "ClearOffset"  },
    { GETPOS,"GETPOS", "GetPosition"},
    { SETPOS,"SETPOS", "SetPosition"},
    { PRCPOS,"PRCPOS", "Where"},
    { DISTNC,"DISTNC", "Distance"},
    { PRIOR ,"PRIOR",  "Priority" },
    { RELATV,"RELATV", "Relative0"},
    { LOOK  ,"LOOK",  "Look"  },
    { RELAT2,"RELAT2", "Relative"},
    { FINDVX,"FINDVX", "FindVertex"},
    { SETVTX,"SETVTX", "SetVertex"},
    { GETVTX,"GETVTX", "GetVertex"},
    { SETPLN,"SETPLN", "SetPlane"},
    { GETPLN,"GETPLN", "GetPlane"},
    { SETSCL,"SETSCL", "SetScale"},
    { GETSCL,"GETSCL", "GetScale"},
    { GRNUM ,"GRNUM",  "gNumber" },
    { GRCHR ,"GRCHR",  "gChar" },
    { GRSTM ,"GRSTM",  "gStringm" },
    { GRSTR ,"GRSTR",  "gString" },
    { GRCOL ,"GRCOL",  "gColor" },
    { GRCLR ,"GRCLR",  "gClear" },
    { GRCSR ,"GRCSR",  "gCursor" },
    { GRPOS ,"GRPOS",  "gPosition" },
    { GRLINE,"GRLINE", "gLine"},
    { GRLINC,"GRLINC", "gLineClear"},
    { SYSTIM,"SYSTIM", "SystemTime"},
    { SYSCNT,"SYSCNT", "SystemCount"},
    { SYSPRC,"SYSPRC", "SystemProcess"},
    { DATA  ,"DATA", "CommonData"  },
    { WRITE ,"WRITE",  "Write" },
    { READ  ,"READ",  "Read"  },
    { LIGHT ,"LIGHT", "Light" },
    { READPL,"READPL", "READPL"},
    { COLNUM,"COLNUM", "NumColors"},
    { COLWDT,"COLWDT", "NumShading"},
    { CNTVTX,"CNTVTX", "VertexCount"},
    { CNTPLY,"CNTPLY", "PolygonCount"},
    { SETCOL,"SETCOL", "SetColor"},
    { RSTCOL,"RSTCOL", "ResetColor"},
    { VER   ,"VER",  "Version"   }
};

int cWord::CheckResWord(const char * Wd, int &sym) {
    unsigned int found = 0;
    unsigned int i = 0;

    while ((!found) && (i < sizeof(ResWord) / sizeof(ResWordType))) {
        if (!strcmp(ResWord[i].wn, Wd)) {
            found = 1;
            sym = ResWord[i].sym;
        }
        i++;
    }
    return found;
}

const char* cWord::GetResWordName(int sym) {
    unsigned int found = 0;
    unsigned int i = 0;
    const char *name = NULL;

    while ((!found) && (i < sizeof(ResWord) / sizeof(ResWordType))) {
        if (ResWord[i].sym == sym) {
            found = 1;
            name = ResWord[i].wn;
        }
        i++;
    }
    return name;
}

int cWord::CheckOpeCode(const char * Wd, int &opc) {
    unsigned int found = 0;
    unsigned int i = 0;

    while ((!found) && (i < sizeof(OpeCode) / sizeof(OPType))) {
        if (!strcmp(OpeCode[i].wn, Wd)) {
            found = 1;
            opc = OpeCode[i].op;
        }
        i++;
    }
    return found;
}

const char* cWord::GetOpecodeName(int opc) {
    unsigned int found = 0;
    unsigned int i = 0;
    const char *name = NULL;

    while ((!found) && (i < sizeof(OpeCode) / sizeof(OPType))) {
        if (OpeCode[i].op == opc) {
            found = 1;
            name = OpeCode[i].opn;
        }
        i++;
    }
    return name;
}

int cWord::CheckRegName(const char * Wd, int &sym) {
    int found = 0;

    if ((Wd[0] == 'R') && (strlen(Wd) == 2)) {
        switch(Wd[1]) {
            case 'Q' : sym = sym_RQ; found = 1; break;
            case 'L' : sym = sym_RL; found = 1; break;
            case 'X' : sym = sym_RX; found = 1; break;
            case 'Y' : sym = sym_RY; found = 1; break;
            case 'Z' : sym = sym_RZ; found = 1; break;
            case 'H' : sym = sym_RH; found = 1; break;
            case 'P' : sym = sym_RP; found = 1; break;
            case 'B' : sym = sym_RB; found = 1; break;
            case '1' : sym = sym_R1; found = 1; break;
            case '2' : sym = sym_R2; found = 1; break;
            case '3' : sym = sym_R3; found = 1; break;
            case '4' : sym = sym_R4; found = 1; break;
            case '5' : sym = sym_R5; found = 1; break;
            case '6' : sym = sym_R6; found = 1; break;
        }
    }
    return found;
}
