/*-------------------------------------------------------------------------*/
/*  J3W ver 5.60  3D Animation Kit  (assembler)                            */
/*  jrlasm.cpp    10/27/2008                                               */
/*  Copyright (C) 1996 - 2008 Jun Mizutani <mizutani.jun@nifty.ne.jp>      */
/*                      All rights reserved.                               */
/*                                                                         */
/*   This file is part of the J3W 3D Animation Kit, and is covered under   */
/*  the terms of the GNU General Public License, version 2. This file has  */
/*  NO WARRANTY. See file COPYING for copyright details.                   */
/*                                                                         */
/*-------------------------------------------------------------------------*/

#include "jrlasm.h"

using namespace std;

cAssembler::cAssembler(cCode *cd) : pCode(cd)
{
}

void cAssembler::asm_error(int eno) {
    cout << '\n' << "File: "<< scan.GetCurrentFileName() << "  Line:"
         << scan.GetLineCount() << "\n";
    cout << scan.GetCurrentLine() << '\n';
    if (eno == 0x99)
        cout << "Error: unknown opecode " << scan.Word() << '\n' ;
    else if (eno == 0x98)
        cout << "Error: LABEL not found " << scan.Word() << '\n' ;
    else if (eno == 0x97)
        cout << "Error: Symbol Definition " << scan.Word() << '\n' ;
    else {
        const char* opn = scan.GetOpecodeName(eno);
        cout << "Error# : " << opn << "  0x" << hex << eno << '\n' ;
    }
    cout << dec;
}

void cAssembler::ShowType(cScan &scan, int type) {
    switch(type) {
        case W_NULL  : cout << "(null ) " << '\n';
            break;
        case W_EOL   : cout << "(eol ) " << '\n';
            break;
        case W_NUM   : cout << "(num  ):" << scan.Word() << '\n';
            break;
        case W_SYM   : cout << "(desig):" << scan.Word() << '\n';
            break;
        case W_REG   : cout << "(reg  ):" << scan.Word() << '\n';
            break;
        case W_CHAR  : cout << "(char ):" << scan.Word() << '\n';
            break;
        case W_STRING: cout << "(str  ):" << scan.Word() << '\n';
            break;
        case W_LABEL : cout << "(label):" << scan.Word() << '\n';
            break;
        case W_OP    : cout << "(opecd):" << hex << scan.Num()
                            << "  " << scan.Word() << dec << '\n';
            break;
    }
}

int cAssembler::asm_line(int op, int pass) {
    int  tc;
    int type;
    int err = 0;

    switch(op) {
        case  NOOP :
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) tc = SetReg1(tc, int(scan.Num()));
            else if (type == W_NUM) tc = SetParam16(tc, int(scan.Num()));
            else err = op;
            pCode->set_code(tc);
            break;
        case  THROW :
            tc = SetOpcode(op);
            pCode->set_code(tc);
            break;
        case  INC:
        case  DEC:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) tc = SetReg1(tc, int(scan.Num()));
            else err = op;
            pCode->set_code(tc);
            break;

        case  ADD  :
        case  SUB  :
        case  MUL  :
        case  DIV  :
        case  SHL  :
        case  SHR  :
        case  AND  :
        case  OR   :
        case  NOT  :
        case  CMP  :
        case  NEG  :
        case  SQRT :
        case  SIN  :
        case  COS  :
        case  ATAN :
        case  RANDOM :
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) {
                tc = SetReg1(tc, int(scan.Num()));
                type = scan.NextWord();
                if (type == W_REG)
                    tc = SetReg2(tc, int(scan.Num()));
                else if (type == W_DATA) {
                    tc = SetReg2(tc, 0x0F);
                    tc = SetParam16(tc, int(scan.Num()));
                } else if ((type == W_NUM) || (type == W_CHAR)) {

                    pCode->set_code(tc);
                    tc = scan.Num();
                } else err = op;
                pCode->set_code(tc);
            } else err = op;
            break;

        case  BRA  :
        case  BEQ  :
        case  BNE  :
        case  BGR  :
        case  BGE  :
        case  BLS  :
        case  BLE  :
        case  BMI  :
        case  BPL  :
        case  LOOP :
        case  CALL :
        case  ENTER:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_NUM) tc = SetParam24(tc, int(scan.Num()));
            else if (type == W_LABEL) {
                int  val = 0;
                int  st;
                if (pass == 2) {
                    if (scan.SymTable->SearchSymName(scan.Word(), val ,st))
                        tc = SetParam24(tc, int(val));
                    else err = 0x98;
                }
            } else err = op;
            pCode->set_code(tc);
            break;

        case  RETURN:
        case  CLEARA:
        case  PUSHA:
        case  POPA:
        case  CLEARG:
        case  PUSHG:
        case  POPG:
        case  LEAVE:

            tc = SetOpcode(op);
            pCode->set_code(tc);
            break;

        case  PUSH :
        case  POP  :
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) tc = SetReg1(tc, int(scan.Num()));
            else err = op;
            pCode->set_code(tc);
            break;

        case  LOAD :
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) {
                tc = SetReg1(tc, int(scan.Num()));
                type = scan.NextWord();
                if (type == W_REG)
                     tc = SetReg2(tc, int(scan.Num()));
                else if (type == W_DATA) {
                    tc = SetReg2(tc, 0x0F);
                    tc = SetParam16(tc, int(scan.Num()));
                } else if ((type == W_NUM) || (type == W_CHAR)) {
                    pCode->set_code(tc);
                    tc = scan.Num();
                } else if (type == W_LABEL) {
                  int  val = 0;
                  int  st;
                  if (pass == 2) {
                      if (scan.SymTable->SearchSymName(scan.Word(), val ,st)) {
                          pCode->set_code(tc);
                          tc = val;
                      } else err = 0x98;
                  } else pCode->set_code(tc);
                } else err = op;
            } else err = op;
            pCode->set_code(tc);
            break;

        case  LOADM :
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_DATA) {
                tc = SetParam16(tc, int(scan.Num()));
                pCode->set_code(tc);
                type = scan.NextWord();
                if (type == W_NUM) {
                    pCode->set_code(scan.Num());
                } else err = op;
            } else err = op;
            break;

        case  CALLTB:
        case  XLOAD:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) {
                tc = SetReg1(tc, int(scan.Num()));
                type = scan.NextWord();
                if (type == W_REG)
                    tc = SetReg2(tc, int(scan.Num()));
                else err = op;
            } else err = op;
            pCode->set_code(tc);
            break;

        case  CLOAD:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) {
                tc = SetReg1(tc, int(scan.Num()));
                type = scan.NextWord();
                if (type == W_REG)
                    tc = SetReg2(tc, int(scan.Num()));
                else if (type == W_NUM)
                    tc = SetParam16(tc, int(scan.Num()));
                else err = op;
            } else err = op;
            pCode->set_code(tc);
            break;

        case  LOADBP :
        case  STORBP :
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) {
                tc = SetReg1(tc, int(scan.Num()));
                tc = SetReg2(tc, 0x00);
                type = scan.NextWord();
                if ((type == W_NUM)||(type == W_DATA)) {
                    tc = SetParam16(tc, int(scan.Num()));
                    pCode->set_code(tc);
                } else err = op;
            } else err = op;
            break;

        case  DEFPUV:
            tc = SetOpcode(op);
            pCode->set_code(tc);
            for (int n=0; n<5; n++) {
                type = scan.NextWord();
                if (type == W_NUM) pCode->set_code(scan.Num());
                else err = op;
            }
            break;

        case  TXFILE:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_NUM) {
                tc = SetReg2(tc, int(scan.Num() & 0xF0)>>4);
                tc = SetReg1(tc, int(scan.Num() & 0x0F));
                type = scan.NextWord();
                if (type == W_STRING) {
                    int n = (scan.Len() >> 2) + 1;
                    tc = SetParam16(tc, n);
                    pCode->set_code(tc);
                    for (int i = 0; i < n; i++) {
                        tc = 0;
                        for (int j = 0; j < 4; j++)
                            tc |= int(scan.GetChar(i*4+j)) << (8*j);
                        pCode->set_code(tc);
                    }
                } else err = op;
            } else err = op;
            break;

        case  TXSCAL:
        case  TXCHG:
        case  TXBIAS:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) {
                tc = SetReg1(tc, int(scan.Num()));
                type = scan.NextWord();
                if (type == W_REG)
                    tc = SetReg2(tc, int(scan.Num()));
                else err = op;
            } else if (type == W_NUM) {
                tc = SetParam16(tc, int(scan.Num()));
                pCode->set_code(tc);
                type = scan.NextWord();
                if (type == W_NUM) tc = scan.Num();
                else err = op;
            } else err = op;
            pCode->set_code(tc);
            break;

        case  STORE :
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) {
                tc = SetReg1(tc, int(scan.Num()));
                tc = SetReg2(tc, 0x0F);
                type = scan.NextWord();
                if (type == W_DATA) {
                    tc = SetParam16(tc, int(scan.Num()));
                    pCode->set_code(tc);
                } else err = op;
            } else err = op;
            break;

        case  XSTORE:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) {
                tc = SetReg1(tc, int(scan.Num()));
                type = scan.NextWord();
                if (type == W_REG)
                    tc = SetReg2(tc, int(scan.Num()));
                else err = op;
            } else err = op;
            pCode->set_code(tc);
            break;

        case  CSTORE:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) {
                tc = SetReg1(tc, int(scan.Num()));
                type = scan.NextWord();
                if (type == W_REG)
                    tc = SetReg2(tc, int(scan.Num()));
                else if (type == W_NUM)
                    tc = SetParam16(tc, int(scan.Num()));
                else err = op;
            } else err = op;
            pCode->set_code(tc);
            break;

        case  GENPRC:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_LABEL) {
                int  val = 0;
                int  st;
                if (pass == 2) {
                    if (scan.SymTable->SearchSymName(scan.Word(), val ,st))
                        tc = SetParam24(tc, int(val));
                    else err = 0x98;
                }
            } else if (type == W_NUM) {
                tc = SetParam24(tc, int(scan.Num()));
            } else err = op;
            pCode->set_code(tc);
            type = scan.NextWord();
            if (type == W_NUM) tc = scan.Num();
            else err = op;
            pCode->set_code(tc);
            break;

        case  DELPRC:
        case  WAIT:
        case  STPALL:
            tc = SetOpcode(op);
            pCode->set_code(tc);
            break;

        case  SEND:
        case  SENDTO:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) {
                tc = SetReg1(tc, int(scan.Num()));
            } else if (type == W_NUM) {
                tc = SetParam16(tc, int(scan.Num()));
            } else err = op;
            type = scan.NextWord();
            if (type == W_REG)
                tc = SetReg2(tc, int(scan.Num()));
            else if (type == W_NUM) {
                pCode->set_code(tc);
                tc = scan.Num();
            } else err = op;
            pCode->set_code(tc);
            break;

        case  RECEIV:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) tc = SetReg1(tc, int(scan.Num()));
            else err = op;
            pCode->set_code(tc);
            break;

        case  CHILD:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_LABEL) {
                int  val = 0;
                int  st;
                if (pass == 2) {
                    if (scan.SymTable->SearchSymName(scan.Word(), val ,st))
                        tc = SetParam24(tc, int(val));
                    else err = 0x98;
                }
            } else if (type == W_NUM) {
                tc = SetParam24(tc, int(scan.Num()));
            } else err = op;
            pCode->set_code(tc);
            type = scan.NextWord();
            if (type == W_NUM) tc = scan.Num();
            else err = op;
            pCode->set_code(tc);
            type = scan.NextWord();
            if (type == W_NUM) {
                pCode->set_code(scan.Num());
                type = scan.NextWord();
                if (type == W_NUM) pCode->set_code(scan.Num());
                else err = op;
            } else err = op;
            break;

        case  TXSET:
        case  TXAXIS:
        case  TXALPH:
        case  TXMAP:
        case  BCOLOR:
        case  ZOOM:
        case  WIRE:
        case  NEAR:
        case  FAR:
        case  EMIT:
        case  PARALL:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) tc = SetReg1(tc, int(scan.Num()));
            else if (type == W_NUM) tc = SetParam16(tc, int(scan.Num()));
            else err = op;
            pCode->set_code(tc);
            break;

        case  INKEY:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) tc = SetReg1(tc, int(scan.Num()));
            else err = op;
            pCode->set_code(tc);
            break;

        case  RAWKEY:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) {
                tc = SetReg1(tc, int(scan.Num()));
                type = scan.NextWord();
                if (type == W_REG)
                  tc = SetReg2(tc, int(scan.Num()));
                else if ((type == W_NUM) || (type == W_CHAR))
                  tc = SetParam16(tc, int(scan.Num()));
                else err = op;
            } else err = op;
            pCode->set_code(tc);
            break;

        case  GRAPHM:
        case  TEXTM:
        case  MOUSE:
            tc = SetOpcode(op);
            pCode->set_code(tc);
            break;

        case  OUTNUM:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) tc = SetReg1(tc, int(scan.Num()));
            else err = op;
            pCode->set_code(tc);
            break;

        case  OUTCHR:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if ((type == W_CHAR) || (type == W_NUM))
                 tc = SetParam16(tc, int(scan.Num()));
            else if (type == W_REG) tc = SetReg1(tc, int(scan.Num()));
            else err = op;
            pCode->set_code(tc);
            break;

        case  OUTSTM:
        case  GRSTM:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) {
                tc = SetReg1(tc, int(scan.Num()));
                tc = SetReg2(tc, 0x00);
                type = scan.NextWord();
                if (type == W_NUM) {
                    tc = SetParam16(tc, int(scan.Num()));
                    pCode->set_code(tc);
                } else err = op;
            } else err = op;
            break;

        case  OUTSTR:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_STRING) {
                int n = (scan.Len() >> 2) + 1;
                tc = SetParam16(tc, n);
                pCode->set_code(tc);
                for (int i = 0; i < n; i++) {
                    tc = 0;
                    for (int j = 0; j < 4; j++) {
                        unsigned int  w = (unsigned char)scan.GetChar(i*4+j);
                        w <<= j*8;
                        tc |= w;
                    }
                    pCode->set_code(tc);
                }
            } else err = op;
            break;

        case  NOTE:
        case  MIDI:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_NUM) {
                tc = SetParam16(tc, int(scan.Num()));
            } else err = op;
            pCode->set_code(tc);
            type = scan.NextWord();
            if (type == W_NUM) tc = scan.Num();
            else err = op;
            type = scan.NextWord();
            if (type == W_NUM) pCode->set_code(tc + scan.Num()*256);
            else err = op;
            break;

        case  MOVF :
        case  MOVB :
        case  MOVL :
        case  MOVR :
        case  MOVU :
        case  MOVD :
        case  ROTH :
        case  ROTP :
        case  ROTB :
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) {
                tc = SetReg1(tc, int(scan.Num()));
                type = scan.NextWord();
                if (type == W_REG)
                    tc = SetReg2(tc, int(scan.Num()));
                else err = op;
            } else if (type == W_NUM) {
                tc = SetParam16(tc, int(scan.Num()));
                pCode->set_code(tc);
                type = scan.NextWord();
                if (type == W_NUM) tc = scan.Num();
                else err = op;
            } else err = op;
            pCode->set_code(tc);
            break;

        case  MOVALL:
        case  MOVALH:
        case  SCALE:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) tc = SetReg1(tc, int(scan.Num()));
            else if (type == W_NUM) tc = SetParam16(tc, int(scan.Num()));
            else err = op;
            pCode->set_code(tc);
            break;

        case  GENOBJ:
            tc = SetOpcode(op);
            pCode->set_code(tc);
            type = scan.NextWord();
            if (type == W_NUM) {
                pCode->set_code(scan.Num());
                type = scan.NextWord();
                if (type == W_NUM) pCode->set_code(scan.Num());
                else err = op;
            } else err = op;
            break;

        case  DEFPNT:
            tc = SetOpcode(op);
            pCode->set_code(tc);
            type = scan.NextWord();
            if (type == W_NUM) {
                pCode->set_code(scan.Num());
                type = scan.NextWord();
                if (type == W_NUM) pCode->set_code(scan.Num());
                else err = op;
                type = scan.NextWord();
                if (type == W_NUM) tc = scan.Num();
                else err = op;
            } else err = op;
            pCode->set_code(tc);
            break;

        case  DEFPLN:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) {
                int m;
                tc = SetReg1(tc, int(scan.Num()));
                type = scan.NextWord();
                if (type == W_NUM) {
                    m = int(scan.Num());
                    tc = SetParam16(tc, m);
                    pCode->set_code(tc);
                    for (int i = 0; i<m ;i++) {
                        type = scan.NextWord();
                        if (type == W_NUM) pCode->set_code(scan.Num());
                        else err = op;
                    }
                } else err = op;
            } else err = op;
            break;

        case  SETACL:
        case  GETACL:
        case  SETVEL:
        case  GETVEL:
        case  SETWVL:
        case  GETWVL:
        case  DELOBJ:
        case  SETEYE:
        case  PNTUV:
        case  POINT:
        case  EYEORG:
        case  TRANSP:
        case  UPDTVP:
        case  RSETVP:
        case  SLIDE :
        case  HOME  :
            tc = SetOpcode(op);
            pCode->set_code(tc);
            break;

        case  SETGRV :
        case  SETREF :
        case  SETAMB :
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) tc = SetReg1(tc, int(scan.Num()));
            else if (type == W_NUM) tc = SetParam16(tc, int(scan.Num()));
            else err = op;
            pCode->set_code(tc);
            break;

        case  GETGRV :
        case  GETVXN :
        case  PLANE  :
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) tc = SetReg1(tc, int(scan.Num()));
            else err = op;
            pCode->set_code(tc);
            break;

        case  GETPOS:
        case  SETPOS:
        case  RELAT2:
        case  FINDVX:
        case  SETVTX:
        case  GETVTX:
        case  SETPLN:
        case  GETPLN:
        case  SETSCL:
        case  GETSCL:
            tc = SetOpcode(op);
            pCode->set_code(tc);
            break;

        case  PRCPOS:
        case  DISTNC:
        case  PRIOR:
        case  RELATV:
        case  LOOK:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) tc = SetReg1(tc, int(scan.Num()));
            else if (type == W_NUM) tc = SetParam16(tc, int(scan.Num()));
            else err = op;
            pCode->set_code(tc);
            break;

        case  GRNUM:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) tc = SetReg1(tc, int(scan.Num()));
            else err = op;
            pCode->set_code(tc);
            break;

        case  GRCHR:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) tc = SetReg1(tc, int(scan.Num()));
            else if ((type == W_CHAR) || (type == W_NUM))
                 tc = SetParam16(tc, int(scan.Num()));
            else err = op;
            pCode->set_code(tc);
            break;

        case  GRSTR:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_STRING) {
                int n = (scan.Len() >> 2) + 1;
                tc = SetParam16(tc, n);
                pCode->set_code(tc);
                for (int i = 0; i < n; i++) {
                    tc = 0;
                    for (int j = 0; j < 4; j++)
                        tc |= int(scan.GetChar(i*4+j)) << (8*j);
                    pCode->set_code(tc);
                }
            } else err = op;
            break;

        case  GRCOL:
        case  GRCSR:
        case  GRLINC:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) tc = SetReg1(tc, int(scan.Num()));
            else if (type == W_NUM) tc = SetParam16(tc, int(scan.Num()));
            else err = op;
            pCode->set_code(tc);
            break;

        case  GRCLR:
        case  GRLINE:
        case  RSTCOL:
        case  VER:
            tc = SetOpcode(op);
            pCode->set_code(tc);
            break;

        case  GRPOS:
        case  SYSTIM:
        case  SYSCNT:
        case  SYSPRC:
        case  LIGHT:
        case  COLNUM:
        case  COLWDT:
        case  CNTVTX:
        case  CNTPLY:
        case  SETCOL:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) tc = SetReg1(tc, int(scan.Num()));
            else err = op;
            pCode->set_code(tc);
            break;

        case  DATA:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_REG) {
                int m;
                tc = SetReg1(tc, int(scan.Num()));
                type = scan.NextWord();
                if (type == W_NUM) {
                    m = int(scan.Num());
                    tc = SetParam16(tc, m);
                    pCode->set_code(tc);
                    int i = 0;
                    while ((i<m) && (err == 0)) {
                        type = scan.NextWord();
                        if (type == W_NUM) pCode->set_code(scan.Num());
                        else err = op;
                        i++;
                    }
                } else err = op;
            } else err = op;
            break;

        case  WRITE:
        case  READ:
        case  READPL:
            tc = SetOpcode(op);
            type = scan.NextWord();
            if (type == W_STRING) {
                int n = (scan.Len() >> 2) + 1;
                tc = SetParam16(tc, n);
                pCode->set_code(tc);
                for (int i = 0; i < n; i++) {
                    tc = 0;
                    for (int j = 0; j < 4; j++)
                        tc |= int(scan.GetChar(i*4+j)) << (8*j);
                    pCode->set_code(tc);
                }
            } else err = op;
            break;

        case  DEFINE:
            type = scan.NextWord();
            if (pass == 1) {
                if (type == W_SYM){
                    char symname[32];

                    strcpy(symname, scan.Word());
                    type = scan.NextWord();
                    if (type == W_NUM)
                        scan.SymTable->RegistSymName(symname,
                                                 scan.Num(), S_SYMBOL);
                    else if (type == W_DATA)
                        scan.SymTable->RegistSymName(symname,
                                                 scan.Num(), S_DATA_ADDR);
                    else if (type == W_CHAR)
                        scan.SymTable->RegistSymName(symname,
                                                 scan.Num(), S_SYMBOL);
                    else {
                        err =  0x97;
                        asm_error(err);
                        type = W_NULL;
                    }
                }
            } else type = scan.NextWord();
            break;

        default: err = 0x99;
    }
    return err;
}

int cAssembler::assemble(char *SourceFileName) {
    int err = 0;
    int type;
    int pass=1;

    char FilePath[128];
    char FileName[128];

    int len = strlen(SourceFileName);
    if (len >= 128) return 1;
    int i = len;
#ifdef WIN32
    while ((SourceFileName[i] != '\\') && (SourceFileName[i] != '/') && (i >= 0))
        i--;
#else
    while ((SourceFileName[i] != '/') && (i >= 0)) i--;
#endif
    i++;
    for(int j=0; j< len-i; j++) FileName[j] = SourceFileName[i+j];
    FileName[len - i] = 0;
    for(int j=0; j < i; j++)  FilePath[j]=SourceFileName[j];
    FilePath[i] = 0;

    scan.SetPath(FilePath);

    while ((pass <= 2) && !err) {
        cout << "       PASS " << pass << '\n';
        if (scan.OpenFile(FileName)) {
            type = W_NULL;
            while ((W_EOF != type)&& !err) {
                while ((W_EOL > (type = scan.NextWord())) && !err) {

                    if (type == W_LABEL) {
                        if (pass == 1)
                            scan.SymTable->RegistSymName(scan.Word(),
                                        pCode->Pos(),S_CODE_ADDR);
                        type = scan.NextWord();
                    }
                    if (type == W_OP) {
                       err = asm_line(int(scan.Num()),pass);
                       if (err) asm_error(err);
                    } else if ((type<W_EOL) && (type!=W_NULL)) {
                        err = 0x99;
                        asm_error(err);
                    }
                }
            }
            if (pass == 1) pCode->clear_code();
        }
        pass++;
    }
    return err;
}
