/*-------------------------------------------------------------------------*/
/*  J3W ver 5.60  3D Animation Kit  (assembler)                            */
/*  jrlscan.cpp   10/27/2008                                               */
/*  Copyright (C) 1996 - 2008 Jun Mizutani <mizutani.jun@nifty.ne.jp>      */
/*                      All rights reserved.                               */
/*                                                                         */
/*   This file is part of the J3W 3D Animation Kit, and is covered under   */
/*  the terms of the GNU General Public License, version 2. This file has  */
/*  NO WARRANTY. See file COPYING for copyright details.                   */
/*                                                                         */
/*-------------------------------------------------------------------------*/

#include <ctype.h>
#include "jrlscan.h"

using namespace std;

struct OPType {
    int  op;
    const char* wn;
};

OPType OpeCode[] = {
        { NOOP, "NOOP"   }, { THROW, "THROW" },
        { INC , "INC"    }, { DEC,   "DEC"   },
        { ADD , "ADD"    }, { SUB ,  "SUB"   }, { MUL , "MUL"    },
        { DIV , "DIV"    }, { SHL ,  "SHL"   }, { SHR , "SHR"    },
        { AND , "AND"    }, { OR  ,  "OR"    }, { NOT , "NOT"    },
        { CMP , "CMP"    }, { NEG ,  "NEG"   }, { SQRT ,"SQRT"   },
        { SIN , "SIN"    }, { COS ,  "COS"   }, { ATAN ,"ATAN"   },
        { RANDOM,"RANDOM"},
        { BRA , "BRA"    }, { BEQ , "BEQ"    }, { BNE , "BNE"    },
        { BGR , "BGR"    }, { BGE , "BGE"    }, { BLS , "BLS"    },
        { BLE , "BLE"    }, { BMI , "BMI"    }, { BPL , "BPL"    },
        { LOOP, "LOOP"   },
        { CALL, "CALL"   }, { RETURN,"RETURN"}, { PUSH, "PUSH"   },
        { POP , "POP"    },
        { CLEARA,"CLEARA"}, { PUSHA, "PUSHA" }, { POPA, "POPA"   },
        { CLEARG,"CLEARG"}, { PUSHG, "PUSHG" }, { POPG, "POPG"   },
        { ENTER, "ENTER" }, { LEAVE, "LEAVE" }, { CALLTB,"CALLTB"},
        { LOAD,  "LOAD"  }, { LOADM ,"LOADM" }, { XLOAD,"XLOAD"  },
        { CLOAD, "CLOAD" }, { LOADBP,"LOADBP"},
            { PNTUV, "PNTUV" }, { DEFPUV,"DEFPUV"},
        { TXFILE,"TXFILE"}, { TXSET, "TXSET" }, { TXAXIS,"TXAXIS"},
        { TXALPH,"TXALPH"}, { TXSCAL,"TXSCAL"}, { TXCHG, "TXCHG" },
        { TXMAP, "TXMAP" }, { TXBIAS,"TXBIAS"},
        { STORE, "STORE" }, { XSTORE,"XSTORE"}, { CSTORE,"CSTORE"},
        { STORBP,"STORBP"},
        { GENPRC,"GENPRC"}, { DELPRC,"DELPRC"}, { WAIT, "WAIT"   },
        { STPALL,"STPALL"}, { SEND,  "SEND"  }, { RECEIV,"RECEIV"},
        { SENDTO,"SENDTO"}, { CHILD, "CHILD" },
        { GRAPHM,"GRAPHM"}, { TEXTM, "TEXTM" }, { BCOLOR,"BCOLOR"},
        { ZOOM,  "ZOOM"  }, { WIRE,  "WIRE"  },
        { NEAR,  "NEAR"  }, { FAR,   "FAR"   },
        { EMIT,  "EMIT"  }, { PARALL,"PARALL"},
        { INKEY, "INKEY" }, { RAWKEY,"RAWKEY"}, { MOUSE, "MOUSE" },
        { OUTNUM,"OUTNUM"}, { OUTCHR,"OUTCHR"}, { OUTSTM,"OUTSTM"},
        { OUTSTR,"OUTSTR"}, { NOTE,  "NOTE"  }, { MIDI,  "MIDI"  },
        { MOVF,  "MOVF"  }, { MOVB,  "MOVB"  }, { MOVL,  "MOVL"  },
        { MOVR,  "MOVR"  }, { MOVU,  "MOVU"  }, { MOVD,  "MOVD"  },
        { ROTH,  "ROTH"  }, { ROTP,  "ROTP"  }, { ROTB,  "ROTB"  },
        { MOVALL,"MOVALL"}, { MOVALH,"MOVALH"}, { SCALE, "SCALE" },
        { SETGRV,"SETGRV"}, { GETGRV,"GETGRV"}, { SETACL,"SETACL"},
        { GETACL,"GETACL"}, { SETVEL,"SETVEL"}, { GETVEL,"GETVEL"},
        { SETWVL,"SETWVL"}, { GETWVL,"GETWVL"},
        { GENOBJ,"GENOBJ"}, { DEFPNT,"DEFPNT"}, { DEFPLN,"DEFPLN"},
        { DELOBJ,"DELOBJ"}, { SETEYE,"SETEYE"}, { POINT, "POINT" },
        { SETREF,"SETREF"}, { SETAMB,"SETAMB"}, { EYEORG,"EYEORG"},
        { GETVXN,"GETVXN"}, { PLANE, "PLANE" }, { TRANSP,"TRANSP"},
        { UPDTVP,"UPDTVP"}, { RSETVP,"RSETVP"}, { SLIDE, "SLIDE" },
        { HOME,  "HOME"  },
        { GETPOS,"GETPOS"}, { SETPOS,"SETPOS"}, { PRCPOS,"PRCPOS"},
        { DISTNC,"DISTNC"}, { PRIOR, "PRIOR" }, { RELATV,"RELATV"},
        { LOOK,  "LOOK"  }, { RELAT2,"RELAT2"},
        { FINDVX,"FINDVX"}, { SETVTX,"SETVTX"}, { GETVTX,"GETVTX"},
        { SETPLN,"SETPLN"}, { GETPLN,"GETPLN"}, { SETSCL,"SETSCL"},
        { GETSCL,"GETSCL"},
        { GRNUM, "GRNUM" }, { GRCHR, "GRCHR" }, { GRSTM, "GRSTM" },
        { GRSTR, "GRSTR" }, { GRCOL, "GRCOL" }, { GRCLR, "GRCLR" },
        { GRCSR, "GRCSR" }, { GRPOS, "GRPOS" }, { GRLINE,"GRLINE"},
        { GRLINC,"GRLINC"},
        { SYSTIM,"SYSTIM"}, { SYSCNT,"SYSCNT"}, { SYSPRC,"SYSPRC"},
        { DATA  ,"DATA"  }, { WRITE, "WRITE" }, { READ,  "READ"  },
        { LIGHT ,"LIGHT" },
        { READPL,"READPL"}, { COLNUM,"COLNUM"}, { COLWDT,"COLWDT"},
        { CNTVTX,"CNTVTX"}, { CNTPLY,"CNTPLY"},
        { SETCOL,"SETCOL"}, { RSTCOL,"RSTCOL"}, { VER,   "VER"   },

        { DEFINE,"DEFINE"}

    };

cScan::cScan() {
    SymTable = new cSymTable(1500);
    WdLen = 0;
    strcpy(LineBuf, "");
    strcpy(CurrentLine, "");
    LineLen = 0;
    LinePtr = 0;
    IncNest = 0;
}

cScan::~cScan() {
    delete SymTable;
}

int cScan::GetLine() {
    do {
        LineLen = 0;
        if (!fin[IncNest].eof())
            fin[IncNest].getline(LineBuf,LINEMAX);
        else {
            fin[IncNest].close();
            IncNest--;
            if (IncNest < 0) return 0;
            else {
                strcpy(CurrentFileName, IncFileName[IncNest]);
                CurrentLineCount = IncLineNumber[IncNest] - 1;
            }
        }
        LineCount++;
        CurrentLineCount++;
        IncLineNumber[IncNest] = CurrentLineCount;
        strcpy(CurrentLine, LineBuf);
        LineLen = strlen(LineBuf);
    } while (LineLen == 0);
    while (LineBuf[0] == '%') {
        if (IncNest < INC_NEST-1) IncNest++;
        else {
            cerr << LineBuf << ": Nesting too deep \n";
            return 0;
        }
        int i = 0;
        while ((LineBuf[i+1] != '%') && (i <LineLen)) {
             IncFileName[IncNest][i] = tolower(LineBuf[i+1]);
             i++;
        }
        IncFileName[IncNest][i] = char(0);

        char tempstr[FILENAME_LEN];
        strcpy(tempstr, SourcePath);
        strcat(tempstr, IncFileName[IncNest]);
        strcpy(IncFileName[IncNest], tempstr);

        fin[IncNest].clear();
        fin[IncNest].open(IncFileName[IncNest]);
        if (!fin[IncNest]) {
            cerr << " Cannot open include file: " << IncFileName[IncNest] << '\n';
            return 0;
        }
        strcpy(CurrentFileName, IncFileName[IncNest]);
        CurrentLineCount = 0;
        IncLineNumber[IncNest] = CurrentLineCount;
        if (!fin[IncNest].eof())
            fin[IncNest].getline(LineBuf,LINEMAX);
        else {
            fin[IncNest].close();
            if (IncNest == 0) return 0;
        }
        LineCount++;
        CurrentLineCount++;
        IncLineNumber[IncNest] = CurrentLineCount;
        LineLen = strlen(LineBuf);
    }

    for (int j=0; j<LineLen; j++)
      if (LineBuf[j] == '\t') LineBuf[j] = ' ';

    int i = 0;
    while ((LineBuf[i] != ';') && (i < LineLen)) i++;
    if (i < LineLen) {
        LineBuf[i] = char(0);
        LineLen = i;
    }
    LinePtr = 0;
    return 1;
}

int cScan::CheckOpeCode(int  &opc) {
    unsigned int found = 0;
    unsigned int i = 0;

    while ((!found) && (i < sizeof(OpeCode) / sizeof(OPType))) {
        if (!strcmp(OpeCode[i].wn, Wd)) {
            found = 1;
            opc = OpeCode[i].op;
        }
        i++;
    }
    return found;
}

const char* cScan::GetOpecodeName(int opc) {
    unsigned int found = 0;
    unsigned int i = 0;
    const char *name = NULL;

    while ((!found) && (i < sizeof(OpeCode) / sizeof(OPType))) {
        if (OpeCode[i].op == opc) {
            found = 1;
            name = OpeCode[i].wn;
        }
        i++;
    }
    return name;
}

int cScan::CheckRegName() {
    int reg_no = 0;

    if ((Wd[0] == 'R') && (WdLen == 2)) {
        switch(Wd[1]) {
            case 'Q' : reg_no = 1; break;
            case 'L' : reg_no = 2; break;
            case 'X' : reg_no = 3; break;
            case 'Y' : reg_no = 4; break;
            case 'Z' : reg_no = 5; break;
            case 'H' : reg_no = 6; break;
            case 'P' : reg_no = 7; break;
            case 'B' : reg_no = 8; break;
            case '1' : reg_no = 9; break;
            case '2' : reg_no = 10; break;
            case '3' : reg_no = 11; break;
            case '4' : reg_no = 12; break;
            case '5' : reg_no = 13; break;
            case '6' : reg_no = 14; break;
            default : reg_no = 0; break;
        }
    }
    return reg_no;
}

int cScan::NextWord() {
    int wtype;

    if (LineLen <= LinePtr) {
        if(!GetLine()) return W_EOF;
        else return W_EOL;
    }
    while ((LineBuf[LinePtr] == ' ') && (LinePtr < LineLen)) LinePtr++;
    int i = 0;
    char c = LineBuf[LinePtr];
    if (c == '\'') {
        wtype = W_CHAR;
        LinePtr++;
        while ((LineBuf[LinePtr] != '\'') && (LinePtr < LineLen))
            Wd[i++] = LineBuf[LinePtr++];
        WNum = int(Wd[0]);
    } else if (c == '"') {
        wtype = W_STRING;
        LinePtr++;
        while ((LineBuf[LinePtr] != '"') && (LinePtr < LineLen))
            Wd[i++] = LineBuf[LinePtr++];
    } else {
        wtype = W_SYM;
        while ((LineBuf[LinePtr] != ' ') && (LinePtr < LineLen))
            Wd[i++] = LineBuf[LinePtr++];
    }
    if (LinePtr < LineLen)  LinePtr++;
    Wd[i] = (char) 0;
    WdLen = strlen(Wd);
    if (wtype == W_SYM) {
        if(WdLen == 0) wtype = W_NULL;
        else if (Wd[WdLen -1] == ':') wtype = W_LABEL;
        else if ((WNum = CheckRegName())) wtype = W_REG;
        else {
            char *endptr;
            int  dataMem = 0;
            if (Wd[WdLen -1] == '#') {
                dataMem = 1;
                Wd[--WdLen] = char(0);
            }
            WNum = strtol(Wd, &endptr, 0);
            if (!*endptr) wtype = W_NUM;
            else wtype = W_SYM;
            if ((wtype == W_NUM) && dataMem) wtype = W_DATA;
        }
    }
    if (wtype == W_SYM) {
        if (CheckOpeCode(WNum)) wtype = W_OP;
    }
    if  (wtype == W_SYM) {
        int st;
        if (SymTable->SearchSymName(Wd, WNum, st)) {
            if (st == S_SYMBOL)    wtype = W_NUM;
            if (st == S_DATA_ADDR) wtype = W_DATA;
        }
    }
    return wtype;
}

int cScan::OpenFile(char *FileName) {
    char fullpath[FILENAME_LEN];

    IncNest = 0;
    LineCount = 0;
    CurrentLineCount = 0;
    strcpy(fullpath, SourcePath);
    strcat(fullpath, FileName);
    fin[IncNest].clear();
    setlocale(LC_ALL, "");
    fin[IncNest].open(fullpath);
    strcpy(CurrentFileName, fullpath);
    strcpy(IncFileName[IncNest], fullpath);
    IncLineNumber[IncNest] = CurrentLineCount;
    if (!fin[IncNest]) {
        cerr << " Cannot open file: " << fullpath << '\n';
        return 0;
    } else return 1;
}

