/*-------------------------------------------------------------------------*/
/*  J3W ver 6.51  3D Animation Kit                                         */
/*  hobj3d.cpp    7/12/2012                                               */
/*  Copyright (C) 1996 - 2012 Jun Mizutani <mizutani.jun@nifty.ne.jp>      */
/*                      All rights reserved.                               */
/*                                                                         */
/*   This file is part of the J3W 3D Animation Kit, and is covered under   */
/*  the terms of the GNU General Public License, version 2. This file has  */
/*  NO WARRANTY. See file COPYING for copyright details.                   */
/*                                                                         */
/*-------------------------------------------------------------------------*/

#include  "hobj3d.h"
#include  "misc3d.h"

THObj3D::THObj3D(int vn, int pn, THObj3D* p):TObject3D(vn,pn),
                           velocity(0,0,0), wvelocity(0,0,0), accele(0,0,0)
{
    gravity  = 0;
    parent   = p;
    brother  = 0;
    child    = 0;
    sx = 1.0;
    sy = 1.0;
    sz = 1.0;
    sf = 0;
}

THObj3D::~THObj3D()
{
    THObj3D *p;
    THObj3D *q;

    if (child) {
        p = child;
        do {
            p->parent = 0;
            p->Axis.set_matrix(p->wm);
            p->Axis.set_origin(p->wp.x, p->wp.y, p->wp.z);
            q = p->brother;
            p->brother = 0;
            p = q;
        } while (p != 0);
    }
    if (parent) {
      if (parent->child == this) {
          if (brother == 0) parent->child = 0;
          else parent->child = brother;
      } else {
          p = parent->child;
          while (p->brother != this) p = p->brother;
          if (brother == 0) p->brother = 0;
          else p->brother = brother;
      }
    }
}

void THObj3D::SetWorld()
{
    wp = Axis.get_origin();
    wm = Axis.get_matrix();
}

void THObj3D::SetNodeWorldMatrix(int  delta)
{
    Vector lg;
    Vector  d;

    if (!parent) {
        double time = delta / 1000.0;
        double t2 = time*time / 2;
        lg.x = 0; lg.y = 0; lg.z = 0;
        if (gravity != 0)
            lg = ConvCoordinate2(wm, Vector(0,0,gravity));
        d.x = iround(((lg.x  + accele.x) * t2 + velocity.x * time)/1000);
        d.y = iround(((lg.y  + accele.y) * t2 + velocity.y * time)/1000);
        d.z = iround(((lg.z  + accele.z) * t2 + velocity.z * time)/1000);
        velocity.x = iround(double(lg.x + accele.x) * time + velocity.x);
        velocity.y = iround(double(lg.y + accele.y) * time + velocity.y);
        velocity.z = iround(double(lg.z + accele.z) * time + velocity.z);
        Axis.move(d);
        wvelocity = ConvCoordinate(wm, velocity);

        int  h = iround(AngleVelocity.x * time);
        int  p = iround(AngleVelocity.y * time);
        int  b = iround(AngleVelocity.z * time);
        Axis.rotate(h, p, b);
        SetWorld();
    } else {

        wp.x = int(Axis.get_origin().x * parent->sx + 0.5);
        wp.y = int(Axis.get_origin().y * parent->sy + 0.5);
        wp.z = int(Axis.get_origin().z * parent->sz + 0.5);

        wp = ConvCoordinate(parent->wm, wp);
        wp = wp + parent->wp;

        wm = MulMatrix3(parent->wm, Axis.get_matrix());

    }

    if (child) child->SetNodeWorldMatrix(delta);

    if (brother) brother->SetNodeWorldMatrix(delta);
}

void THObj3D::SetVelocity(Vector v)
{
    velocity = v;
    wvelocity = ConvCoordinate(wm, v);
}

void THObj3D::SetWVelocity(Vector wv)
{
    wvelocity = wv;
    velocity = ConvCoordinate2(wm, wv);
}

void THObj3D::show(THObj3D *eye)
{
    Vector v, u;
    Matrix3 m, e;

    e = eye->wm;
    m.m1 = e.m7 * wm.m7 + e.m4 * wm.m4 + e.m1 * wm.m1;
    m.m2 = e.m7 * wm.m8 + e.m4 * wm.m5 + e.m1 * wm.m2;
    m.m3 = e.m7 * wm.m9 + e.m4 * wm.m6 + e.m1 * wm.m3;
    m.m4 = e.m8 * wm.m7 + e.m5 * wm.m4 + e.m2 * wm.m1;
    m.m5 = e.m8 * wm.m8 + e.m5 * wm.m5 + e.m2 * wm.m2;
    m.m6 = e.m8 * wm.m9 + e.m5 * wm.m6 + e.m2 * wm.m3;
    m.m7 = e.m9 * wm.m7 + e.m6 * wm.m4 + e.m3 * wm.m1;
    m.m8 = e.m9 * wm.m8 + e.m6 * wm.m5 + e.m3 * wm.m2;
    m.m9 = e.m9 * wm.m9 + e.m6 * wm.m6 + e.m3 * wm.m3;
    v = wp - eye->wp;
    u.x = iround(e.m1 * v.x + e.m4 * v.y + e.m7 * v.z);
    u.y = iround(e.m2 * v.x + e.m5 * v.y + e.m8 * v.z);
    u.z = iround(e.m3 * v.x + e.m6 * v.y + e.m9 * v.z);

    int n =  Polygon.Vertex.GetCount();
    for (int i=0; i<n; i++) {
        Vector w = Polygon.Vertex.vert[i]->local;
        if (sf) {
            w.x = int(w.x * sx + 0.5);
            w.y = int(w.y * sy + 0.5);
            w.z = int(w.z * sz + 0.5);
        }
        v = ConvCoordinate(m, w);
        Polygon.Vertex.vert[i]->eye = v + u;
    }

    n = Polygon.GetCount();
    for (int i=0; i<n; i++) { /* convert normal vector */
        if (Polygon.pg[i]->f_normal)
            Polygon.pg[i]->enormal = ConvCoordinate_d(m, Polygon.pg[i]->normal);
    }
}

void THObj3D::GetWorldPosition(Vector &pos, int  &h, int  &p, int  &b)
{
    pos = wp;
    b  = DEG( atan2(wm.m8, wm.m9) );
    p  = DEG( - asin(wm.m7) );
    h  = DEG( atan2(wm.m4, wm.m1) );
}

Vector THObj3D::GetWorldVertexPosition(int i)
{
    Vector v;

    int n =  Polygon.Vertex.GetCount();
    if ((i>=0) && (i<n)) {
        Vector w = Polygon.Vertex.vert[i]->local;
        if (sf) {
            w.x = int(w.x * sx + 0.5);
            w.y = int(w.y * sy + 0.5);
            w.z = int(w.z * sz + 0.5);
        }
        v = ConvCoordinate(wm, w);
    }
    return v + wp;
}
