#!/usr/bin/luajit
-- ---------------------------------------------
-- jsCommand.lua  2015/02/28
--   Copyright (c) 2015 Jun Mizutani,
--   released under the MIT open source license.
-- ---------------------------------------------

local ffi = require("ffi")
local bit = require("bit")
local bnot, bor, band = bit.bnot, bit.bor, bit.band

ffi.cdef[[
  struct js_event {
    unsigned int time;  /* event timestamp in milliseconds */
    short value;        /* value */
    char  type;         /* event type */
    char  number;       /* axis/button number */
  };

  static const int F_SETFL = 4;

  int ioctl(int, unsigned long, ...);
  int open(const char* filename, int flags);
  int read(int fd, void *buf, unsigned int nbytes);
  int fcntl(int fd, int cmd, ...);
  int close(int fd);
  int poll(struct pollfd *fds, unsigned long nfds, int timeout);
]]

local JS_EVENT_BUTTON = 0x1
local JS_EVENT_AXIS   = 0x2
local JS_EVENT_INIT   = 0x80
local JSIOCGVERSION   = 0x80046a01
local JSIOCGAXES      = 0x80016a11
local JSIOCGBUTTONS   = 0x80016a12
local JSIOCGNAME      = 0x80006a13

local EVENT_CLEAR_MSEC   = 1000

local devices = {}
local initialized = false

function init()
  for i = 0, 7 do
    openJoystick(i)
  end
  if #devices > 0 then
    setDeviceInfo()
    initialized = true
  end
  return #devices
end

function open(device)
  local O_RDONLY   = 0
  local O_NONBLOCK = 0x800
  local fd = ffi.C.open(device, O_RDONLY + O_NONBLOCK)
  return fd
end

function openJoystick(n)
  if (n >= 0) and (n <=7) then
    local fd = open("/dev/input/js" .. tonumber(n))
    if fd >= 0 then
      local device = {}
      device.num = n
      device.fd = fd
      table.insert(devices, device)
      devices[n + 1].last_time = 0
      devices[n + 1].event_list = {}
      devices[n + 1].eventStr = {}
      return fd
    else
      return -1
    end
  else
    return -1
  end
end

function setDeviceInfo()
  local version = ffi.new("int[1]")
  local axes = ffi.new("unsigned char[1]")
  local buttons = ffi.new("unsigned char[1]")
  local name = ffi.new("char[128]")
  for i = 1, #devices do
    local fd = devices[i].fd
    ffi.C.ioctl(fd, JSIOCGVERSION, version)
    ffi.C.ioctl(fd, JSIOCGAXES, axes)
    ffi.C.ioctl(fd, JSIOCGBUTTONS, buttons)
    ffi.C.ioctl(fd, JSIOCGNAME + 128 * 0x10000, name)
    devices[i].version = version[0]
    devices[i].num_axes = axes[0]
    devices[i].num_buttons = buttons[0]
    devices[i].name = ffi.string(name)
    devices[i].axes = {}
    devices[i].buttons = {}
    for j = 1, axes[0] do
      table.insert(devices[i].axes,
                   {type = 0, number = 0, value = 0, time = 0})
    end
    for j = 1, buttons[0] do
      table.insert(devices[i].buttons,
                   {type = 0, number = 0, value = 0, time = 0})
    end
  end
end

function readOneEvent(device)
  if (device < 1) or (device > #devices) then
    return nil -- invalid device
  end
  local fd = devices[device].fd
  local js = ffi.new("struct js_event[1]")
  local size = ffi.sizeof(js)
  local res = ffi.C.read(fd, js, size)
  if res == size then
    local event = js[0]
    return event
  else
    return nil
  end
end

function recordEventStr(device, event)
  local js = devices[device]
  local n = event.number
  if event.type == JS_EVENT_BUTTON then
    if n == 0 then      table.insert(js.eventStr, "A")
    elseif n == 1  then table.insert(js.eventStr, "B")
    elseif n == 2  then table.insert(js.eventStr, "X")
    elseif n == 3  then table.insert(js.eventStr, "Y")
    elseif n == 4  then table.insert(js.eventStr, "L")
    elseif n == 5  then table.insert(js.eventStr, "R")
    elseif n == 6  then table.insert(js.eventStr, "E")
    elseif n == 7  then table.insert(js.eventStr, "S")
    end
  elseif event.type == JS_EVENT_AXIS then
    local v = event.value
    if n == 0 then
      if v > 0 then     table.insert(js.eventStr, "r")
      elseif v < 0 then table.insert(js.eventStr, "l")
      end
    elseif n == 1 then
      if v > 0 then     table.insert(js.eventStr, "d")
      elseif v < 0 then table.insert(js.eventStr, "u")
      end
    end
  end
end

function clearEventStr(device)
  devices[device].eventStr = {}
end

function getEventStr(device)
  return table.concat(devices[device].eventStr)
end

function readAllEvents(device)
  local event
  local numEvent = 0
  local js = devices[device]
  repeat
    event = readOneEvent(device)
    if event ~= nil then
      numEvent = numEvent + 1
      if (event.time - js.last_time) > EVENT_CLEAR_MSEC then
        clearEventStr(device)
      end
      local num = event.number + 1
      if band(event.type, JS_EVENT_INIT) > 0 then
        js.start = event.time;
        event.type = band(event.type, bnot(JS_EVENT_INIT))
        if event.type == JS_EVENT_BUTTON then
          js.buttons[num].value = event.value
          js.buttons[num].time = event.time
        else
          js.axes[num].value = event.value
          js.axes[num].time = event.time
        end
      elseif event.type == JS_EVENT_BUTTON then
        local button =  js.buttons[num]
        if event.value == 0 then
          button.longPush = event.time - button.time
          button.longRelease = 0
        else
          button.longPush = 0
          button.longRelease = event.time - button.time
          recordEventStr(device, event)
        end
        button.value = event.value
        button.time = event.time
      else
        js.axes[num].value = event.value
        js.axes[num].time = event.time
        if event.value ~= 0 then
          recordEventStr(device, event)
        end
      end
      js.last_time = event.time
    end
  until event == nil
  return numEvent
end

function readAllDevices()
  for i = 1, #devices do
    readAllEvents(i)
  end
end

function exec(command)
  local f = io.popen(command .. " 2>&1", "r")
  if f ~= nil then
    local result = f:read('*a')  -- read output
    f:close()
    return result
  else
    return "execute error"
  end
end

function initCommandList()
  commandList = { BABABA = "help" }
end

function addCommand(keys, command)
  commandList[keys] = command
  return #commandList
end

function listCommand(f)
  f:write("[A] [B] [X] [Y] [L] [R] [Start] [sElect], ")
  f:write("up, down, left, right\n")
  for key, command in pairs(commandList) do
    f:write(key .. " : " .. command .. "\n")
  end
end

function execCommand(numDevice, str, f)
  for key, command in pairs(commandList) do
    if str == key then
      if command == "quit" then os.exit() end
      if command == "help" then
        listCommand(f)
        clearEventStr(numDevice)
      else
        f:write("\nkey = ".. key .. ", command = " .. command .. "\n")
        local result = exec(command)
        if result ~= nil then f:write(result) end
        clearEventStr(numDevice)
      end
    end
  end
end

function checkCommand(numDevice, f)
  local numEvent = readAllEvents(numDevice)
  if numEvent > 0 then
    str = getEventStr(numDevice)
    execCommand(numDevice, str, f)
    return str
  end
  return nil
end

function getNoOfDevices()
  return #devices
end

-- device_num: 1..8
function getName(device_num)
  return devices[device_num].name
end

function getVersion(device_num)
  return devices[device_num].version
end

function getNoOfAxes(device_num)
  return devices[device_num].num_axes
end

function getNoOfButtons(device_num)
  return devices[device_num].num_buttons
end

function sleep(sec)
  ffi.C.poll(nil, 0, sec * 1000)
end

function printf(f, fmt, ...)
  f:write(string.format(fmt, ...))
end

function deviceList(f)
  local num_device = getNoOfDevices()
  for i = 1, num_device do
    printf(f, "Name: %s\n", getName(i))
    printf(f, "Ver.: %s\n", getVersion(i))
    local num_axes = getNoOfAxes(i)
    printf(f, "  No. of Axes    : %d\n", num_axes)
    local num_buttons = getNoOfButtons(i)
    printf(f, "  No. of Buttons : %d\n", num_buttons)
  end
  printf(f, "\nhelp : BABABA\n\n")
end

-- -------------
--     main
-- -------------
  local numOfDevices = init()
  local str
  local f
  if arg[1] == "-d" then DEBUG = true end
  if arg[1] == "-c" then CONSOLE = true end
  if CONSOLE then
    f = io.open("/dev/console", "w")
  else
    f = io.open("/dev/stdout", "w")
  end

  initCommandList()
 
  -- A, B, X, Y, L, R, Start, sElect, up, down, left, right
  addCommand("ESESESES", "quit")
  addCommand("uuddlrlrBA", "/sbin/shutdown -h now")
  addCommand("uuddlrlrES", "/sbin/reboot")
  addCommand("BBBB", "/bin/date -R")
  addCommand("SSSS", "/usr/bin/pstree -A")
  addCommand("SESE", "/bin/sync")
  addCommand("EEEE", "/bin/ps aux")
  addCommand("LLLL", "/sbin/lsmod")
  addCommand("lllr", "/sbin/lsmod")
  addCommand("RRRR", "/usr/bin/who -a")
  addCommand("rrrl", "/usr/bin/who -a")
  addCommand("rrrr", "/bin/netstat -a")
  addCommand("llrr", "/sbin/ifconfig -a")
  addCommand("llll", "/usr/bin/vmstat -s")
  addCommand("lrlr", "/usr/bin/uptime")
  addCommand("uuuu", "/bin/cat /sys/class/thermal/thermal_zone0/temp")
  addCommand("dddd",
    "/bin/cat /sys/devices/system/cpu/cpu0/cpufreq/scaling_cur_freq")
  addCommand("dudu", "/bin/df -kT")
  addCommand("uldr", "/bin/uname -a")
  addCommand("Eddd", "/usr/bin/env")
  --
  if arg[1] == "-h" then
    listCommand(f)
    os.exit()
  end

  if numOfDevices > 0 then
    deviceList(f)
    while (true) do
      str = checkCommand(1, f)
      if DEBUG and str ~= nil then io.write(str.."\n") end
      sleep(0.5)
    end
  end
